//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file KBRTestApp.cc
 * @author Bernhard Heep
 */

#include <IPAddressResolver.h>
#include <CommonAPIMessages_m.h>

#include "KBRTestApp.h"
#include "KBRTestMessage_m.h"


Define_Module(KBRTestApp);

void KBRTestApp::initializeApp(int stage)
{
    if(stage != MIN_STAGE_APP)
        return;

    lookupNodeIds = par("lookupNodeIds");
    mean = par("messageDelay");
    deviation = mean / 10;
    activeNetwInitPhase = par("activeNetwInitPhase");

    numDelivered = 0;
    bytesDelivered = 0;
    WATCH(numDelivered);
    WATCH(bytesDelivered);

    delayVector.setName("Delay Time");
    hopCountVector.setName("Hop Count");

    // initiate test message emision
    cMessage* test_timer = new cMessage("test_timer");
    scheduleAt(simulation.simTime() + truncnormal(mean, deviation), test_timer);
}

void KBRTestApp::handleTimerEvent(cMessage* msg)
{
    if (msg->isName("test_timer")) {
        // schedule next timer event
        scheduleAt(simulation.simTime() + truncnormal(mean, deviation), msg);

        // do nothing if the network is still in the initialization phase
        if((!activeNetwInitPhase) && (underlayConfigurator->isInit()))
            return;
        OverlayKey destKey;
        if ( lookupNodeIds ) {
            destKey = bootstrapOracle->getBootstrapNode().key;
            // do nothing if there are currently no nodes in the network
            if (destKey.isUnspecified())
                return;
        } else {
            // generate random destination key
            destKey = OverlayKey::random();
        }

        // create a 100 byte test message
        KBRTestMessage* testMsg = new KBRTestMessage("KBRTestMessage");
        testMsg->setId(id());
        testMsg->setSeqNum(numSent);
        testMsg->setByteLength(100);

	globalStatistics->sentKBRTestAppMessages++;

        callRoute(destKey, testMsg);
    }
}

void KBRTestApp::deliver(OverlayKey& key, cMessage* msg)
{
    KBRTestMessage* testMsg = check_and_cast<KBRTestMessage*>(msg);
    OverlayCtrlInfo* overlayCtrlInfo =
        check_and_cast<OverlayCtrlInfo*>(msg->removeControlInfo());

    // Return statistical data to the sender.
    if(cModule* mod = simulation.module(testMsg->getId())) {
        if(KBRTestApp* sender = dynamic_cast<KBRTestApp*>(mod)) {
            if((!lookupNodeIds) || (overlayCtrlInfo->getThisNode().key ==
                                    overlayCtrlInfo->getDestKey())) {
                sender->evaluateData((simTime() - testMsg->creationTime()),
                                     overlayCtrlInfo->getHopCount(),
                                     testMsg->byteLength());
            }
        }
    }

    EV << "(KBRTestAPP) received \"" << testMsg->name()
    << "\" (seqNr: " << testMsg->getSeqNum()
    << ")\n             with destination key: " << key.toString(16)
    << "\n             (thisNode.key = "
    << overlayCtrlInfo->getThisNode().key.toString(16) << ")"
    << endl;

    delete overlayCtrlInfo;
    delete testMsg;
}

void KBRTestApp::evaluateData(simtime_t timeDelay, int hopCount, long int bytes)
{
    // count the number and size of successfully delivered messages
    numDelivered++;
    bytesDelivered += bytes;
    globalStatistics->deliveredKBRTestAppMessages++;


    // record vectorial data
    delayVector.record(timeDelay);
    hopCountVector.record(hopCount);
    globalStatistics->globalDelayVector.record(timeDelay);
}

void KBRTestApp::finishApp()
{
    recordScalar("KBRTestApp: Delivered Messages", numDelivered);
    recordScalar("KBRTestApp: Delivered Bytes", bytesDelivered);

    globalStatistics->testCount++;


    if (numSent != 0)
        recordScalar("KBRTestApp: Delivery Ratio",
		     (float) numDelivered / (float) numSent);
    else
        recordScalar("KBRTestApp: Delivery Ratio", 0);
}

