//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseApp.cc
 * @author Bernhard Heep
 */

#include <IPAddressResolver.h>
#include <CommonAPIMessages_m.h>

#include "BaseApp.h"
#include "BootstrapOracleAccess.h"
#include "GlobalStatisticsAccess.h"
#include "UnderlayConfiguratorAccess.h"


BaseApp::~BaseApp()
{
    //...
}

int BaseApp::numInitStages() const
{
    return MAX_STAGE_APP + 1;
}

void BaseApp::initialize(int stage)
{
    if(stage == MIN_STAGE_APP) {
        // fetch parameters
        debugOutput = par("debugOutput");
        onlyCommonAPIMessages = true;

        bootstrapOracle = BootstrapOracleAccess().get();
        underlayConfigurator = UnderlayConfiguratorAccess().get();
        globalStatistics = GlobalStatisticsAccess().get();

        // determine the terminal's node handle
        thisNode.ip = IPAddressResolver().addressOf(parentModule()->parentModule()).get4();
        //thisNode.key = //solution?

        // statistics
        numSent = 0;
        numReceived = 0;
        bytesSent = 0;
        bytesReceived = 0;

        WATCH(numSent);
        WATCH(numReceived);
        WATCH(bytesSent);
        WATCH(bytesReceived);
    }
    if(stage >= MIN_STAGE_APP && stage <= MAX_STAGE_APP)
        initializeApp(stage);
}

void BaseApp::initializeApp(int stage)
{
    // ...
}

// Process messages passed up from the overlay.
void BaseApp::handleMessage(cMessage* msg)
{
    // Process self-messages.
    if(msg->isSelfMessage()) {
	handleTimerEvent(msg);
	return;
    }
    
    if(msg->arrivedOn("from_overlay")) {
	// common API
	if(dynamic_cast<CommonAPIMessage*>(msg) != NULL) {
	    cMessage* tempMsg = msg->decapsulate();
	    // process interface control information
	    OverlayCtrlInfo* overlayCtrlInfo =
		check_and_cast<OverlayCtrlInfo*>(msg->removeControlInfo());
	    tempMsg->setControlInfo(overlayCtrlInfo); //bad solution!

	    if(debugOutput)
		EV << "(BaseApp) Node " << thisNode.ip << " received message "
		   << " (sent to " << overlayCtrlInfo->getDestKey() << ") from node "
		   << overlayCtrlInfo->getSrcNode().ip << "." << endl;

	    if(dynamic_cast<KBRdeliver*>(msg) != NULL) {
		numReceived++;
		bytesReceived += tempMsg->byteLength();
		deliver(overlayCtrlInfo->getDestKey(), tempMsg);
	    }
	    else if(dynamic_cast<KBRforward*>(msg) != NULL)
		forward(overlayCtrlInfo->getDestKey(), tempMsg, NULL);

	    else
		delete tempMsg;

	    delete msg;
	} else if (onlyCommonAPIMessages == false) {
	    numReceived++;
	    bytesReceived += msg->byteLength();
	    handleAppMessage(msg);
	}
    } else if(msg->arrivedOn("from_upperTier")) {
	handleUpperMessage(msg);
    }
}

void BaseApp::handleTimerEvent(cMessage* msg)
{
    // process self-messages
}

void BaseApp::callRoute(const OverlayKey& key, cMessage* msg, const NodeHandle& hint)
{
    // add some interface control information and send the message
    OverlayCtrlInfo* overlayCtrlInfo = new OverlayCtrlInfo();
    overlayCtrlInfo->setDestKey(key);
    overlayCtrlInfo->setHint(hint);

    // create route-message (common API)
    KBRroute* routeMsg = new KBRroute();
    routeMsg->setControlInfo(overlayCtrlInfo);
    routeMsg->encapsulate(msg);

    send(routeMsg, "to_overlay");

    // debug output
    if (debugOutput)
        EV << "(BaseApp::callRoute()) Node " << thisNode.ip << " sent message "
        << id() << "-" << numSent <<  " to destination key "
        << key.toString(16) << "." << endl;

    // count
    numSent++;
    bytesSent += msg->byteLength();
}

void BaseApp::deliver(OverlayKey& key, cMessage* msg)
{
    // deliver...

    delete msg;
}

void BaseApp::forward(OverlayKey&, cMessage* msg, NodeHandle* hint)
{
    // forward...
}

void BaseApp::handleUpperMessage(cMessage* msg)
{
    delete msg;
}

void BaseApp::handleAppMessage(cMessage* msg)
{
    delete msg;
}

void BaseApp::sendMessageToOverlay(cMessage* msg)
{
    numSent++;
    bytesSent += msg->byteLength();

    send(msg, "to_overlay");
}

void BaseApp::finish()
{
    // record scalar data
    recordScalar("BaseApp: Sent Messages", numSent);
    recordScalar("BaseApp: Received Messages", numReceived);
    recordScalar("BaseApp: Sent Bytes", bytesSent);
    recordScalar("BaseApp: Received Bytes", bytesReceived);

    finishApp();
}

void BaseApp::finishApp()
{
    // ...
}
