//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseApp.h
 * @author Bernhard Heep
 */

#ifndef __BASEAPP_H_
#define __BASEAPP_H_

#include <omnetpp.h>

#include <BootstrapOracle.h>
#include <GlobalStatistics.h>
#include <UnderlayConfigurator.h>
#include <NodeHandle.h>
#include <OverlayKey.h>
#include <InitStages.h>


/**
 * Base class for applications (Tier 1)
 *
 * Base class for applications (Tier 1) that use overlay functionality.
 * provides common API for structured overlays (KBR)
 *
 * @see KBRTestApp
 * @author Bernhard Heep
 */
class BaseApp : public cSimpleModule
{
private:

protected:
    /**
     * method to set InitStage
     */
    int numInitStages() const;

    /**
     * initializes base class-attributes
     *
     * @param stage the init stage
     */
    void initialize(int stage);

    /**
     * initializes derived class-attributes
     *
     * @param stage the init stage
     */
    virtual void initializeApp(int stage);

    /**
     * checks for message type and calls corresponding method
     *
     * checks for message type (from overlay or selfmessage) and calls
     * corresponding method
     * like deliver(), forward(), and timer()
     * @param msg the handled message
     */
    void handleMessage(cMessage* msg);

    /**
     * collects statistical data
     */
    void finish();

    /**
     * collects statistical data of derived app
     */
    virtual void finishApp();

    // common API for structured p2p-overlays
    /**
     * calls route-method in overlay
     *
     * encapsulates msg into KBRroute message and sends it to the overlay
     * module
     *
     * @param key destination key
     * @param msg message to route
     * @param hint next hop (usually unused)
     */
    void callRoute(const OverlayKey& key, cMessage* msg,
		   const NodeHandle& hint = NodeHandle::UNSPECIFIED_NODE);

    /**
     * handles delivered messages from overlay
     *
     * method to handle decapsulated KBRdeliver messages from overlay module,
     * should be overwritten in derived application
     * @param key destination key
     * @param msg delivered message
     */
    virtual void deliver(OverlayKey& key, cMessage* msg);

    /**
     * handles messages from overlay to be forwarded
     *
     * method to handle decapsulated KBRdeliver messages from overlay module,
     * should be overwritten in derived application if needed
     * @param key destination key
     * @param msg message to forward
     * @param hint next hop (usually unused)
     */
    virtual void forward(OverlayKey& key, cMessage* msg,
			 NodeHandle* hint = NULL);

    /**
     * processes self-messages
     *
     * method to handle self-messages
     * should be overwritten in derived application if needed
     * @param msg self-message
     */
    virtual void handleTimerEvent(cMessage* msg);

    /**
     * method to handle non-commonAPI messages from the overlay
     *
     * @param msg message to handle
     */
    virtual void handleAppMessage(cMessage* msg);

    virtual void handleUpperMessage(cMessage* msg);

    /**
     * sends non-commonAPI message to the overlay
     *
     * @param msg message to send
     */
    void sendMessageToOverlay(cMessage* msg);

    NodeHandle thisNode; /**< NodeHandle to this node*/
    UnderlayConfigurator* underlayConfigurator; /**< pointer to
						   UnderlayConfigurator
						   in this node*/
    BootstrapOracle* bootstrapOracle; /**< pointer to BootstrapOracle
					 in this node*/

    GlobalStatistics* globalStatistics; /**< pointer to GlobalStatistics module
					 in this node*/


    // parameters
    bool debugOutput; /**< debug output yes/no?*/
    bool onlyCommonAPIMessages; /**< process/send only commonAPI messages? */

    // statistics
    int numSent; /**< number of sent packets*/
    int bytesSent; /**< number of sent bytes*/
    int numReceived; /**< number of received packets*/
    int bytesReceived; /**< number of received bytes*/

public:

    /**
     * virtual destructor
     */
    virtual ~BaseApp();
};

#endif
