//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseLookup.cc
 * @author Sebastian Mies
 */

#include "BaseLookup.h"
#include <iostream>

using namespace std;

//----------------------------------------------------------------------------

LookupListener::~LookupListener()
{}

//----------------------------------------------------------------------------

AbstractLookup::~AbstractLookup()
{}



//----------------------------------------------------------------------------
//- Construction & Destruction -----------------------------------------------
//----------------------------------------------------------------------------
BaseLookup::BaseLookup( BaseOverlay* overlay,
			const BaseLookupConfiguration& config )
{
    this->config = config;
    if (config.secure)
        cout << "WARNING: Secure BaseLookup is still under development!!"
	     << endl;

    this->overlay = overlay;
    this->running = false;
    this->finished = false;
    this->success = false;
}

BaseLookup::~BaseLookup()
{
    stop();
    overlay->removeLookup(this);
}

void BaseLookup::start()
{
    // init params
    this->successfulPaths = 0;
    this->finishedPaths   = 0;
    this->accumulatedHops = 0;

    // init flags
    this->finished = false;
    this->success  = false;
    this->running  = true;

    // init neighborhood vector
    neighbors = NodeVector( numNeighbors == 0 ? 1 : numNeighbors, this );

    // get local closest nodes
    FindNodeCall* call = new FindNodeCall("FindNodeCall");
    NodeVector* nextHops = overlay->findNode(key, call);

    // if this node is new and no nodes are known -> stop lookup
    if (nextHops->size() == 0) {
        stop();
        delete nextHops;
        delete call;
        return;
    }

    // remove find node extensions
    cMessage* findNodeExt = NULL;
    if (call->hasObject("findNodeExt"))
        findNodeExt = (cMessage*)call->removeObject("findNodeExt");
    delete call;

    // distribution of nodes to paths
    uint n = nextHops->size() / config.parallelPaths;

    // not enough nodes for all paths? -> reduce number of parallel paths
    if ( n == 0 ) {
        config.parallelPaths = nextHops->size();
        n = 1;
    }

    // create parallel paths
    int j=0;
    for (int i=0; i<config.parallelPaths; i++) {

        // create state
        BasePathLookup* pathLookup = new BasePathLookup( this );
        paths.push_back( pathLookup );

        // populate next hops
        for ( uint k=0; k<n; k++, j++ )
            pathLookup->add( (*nextHops)[j] );

        // send initial rpcs
        pathLookup->sendRpc( config.parallelRpcs, findNodeExt );
    }
    delete nextHops;
    delete findNodeExt;
}

void BaseLookup::stop()
{
    // only stop if running
    if (!running)
        return;

    // cancel pending rpcs
    for (RpcInfoMap::iterator i = rpcs.begin(); i != rpcs.end(); i++)
        overlay->cancelRpcMessage( i->second.nonce );
    rpcs.clear();

    // delete path lookups
    for (uint i=0; i<paths.size(); i++)
        delete paths[i];
    paths.clear();

    // reset running flag
    running  = false;
    finished = true;

    // inform listener
    if ( listener != NULL )
        listener->lookupFinished(this);
}

inline void BaseLookup::checkStop()
{
    // check if there are rpcs pending or lookup finished
    if ( (successfulPaths >=1 && numNeighbors == 0) ||
            (finishedPaths ==
	     (uint)config.parallelPaths && numNeighbors > 0) ) {

        for (uint i=0; i<paths.size();i++)
            success |= paths[i]->success;
        delete this;

    } else if (rpcs.size() == 0)
        delete this;

}

//----------------------------------------------------------------------------
//- Enhanceable methods ------------------------------------------------------
//----------------------------------------------------------------------------
BasePathLookup* BaseLookup::createPathLookup()
{
    return new BasePathLookup(this);
}


//----------------------------------------------------------------------------
//- Base configuration and state ---------------------------------------------
//----------------------------------------------------------------------------
//virtual public
int BaseLookup::compare( const OverlayKey& lhs, const OverlayKey& rhs ) const
{
    return overlay->distance( key, lhs ).compareTo( overlay->distance(key,
								      rhs));
}


//----------------------------------------------------------------------------
//- Neighbors and visited nodes management -----------------------------------
//----------------------------------------------------------------------------
bool BaseLookup::addNeighbor( const NodeHandle& handle )
{
    bool result = false;

    if (numNeighbors == 0) {
        if (handle.key == key) {
            neighbors.clear();
            neighbors.push_back( handle );
            result = true;
        }
    } else {
        if (config.parallelPaths==1) {
            result = true;
            this->neighbors.push_back(handle);
        } else {
            result = this->neighbors.add(handle);
        }
    }

    return result;
}

void BaseLookup::setVisited( const NodeHandle& handle, bool visited )
{
    if (visited)
        this->visited.insert( handle );
    else
        this->visited.erase( handle );
}

bool BaseLookup::getVisited( const NodeHandle& handle )
{
    return this->visited.count(handle) != 0;
}


//----------------------------------------------------------------------------
//- Parallel RPC distribution ------------------------------------------------
//----------------------------------------------------------------------------
void BaseLookup::handleRpcResponse( BaseResponseMessage* msg,
				    int rpcId, simtime_t rtt )
{
    // check flags
    if (finished || !running)
        return;

    // get source, cast messages and mark node as visited
    const NodeHandle& src = msg->getSrcNode();
    FindNodeResponse* findNodeResponse = dynamic_cast<FindNodeResponse*>(msg);
    PingResponse* pingResponse = dynamic_cast<PingResponse*>(msg);
    setVisited( src );

    if ( findNodeResponse != NULL || pingResponse != NULL) {
        // add authentificated neighbor
//        addNeighbor( src );
    }

    // handle find node response
    if ( findNodeResponse != NULL ) {

        // check if rpc info is available, no -> exit
        if (rpcs.count(src)==0)
            return;

        // get info
        RpcInfoVector infos = rpcs[src];
        rpcs.erase(src);

        // add to neighborlist if not secure
  //      if (!config.secure)
  //          for (uint i=0; i<findNodeResponse->getClosestNodesArraySize(); i++)
  //              addNeighbor( findNodeResponse->getClosestNodes(i) );

        // iterate
        bool rpcHandled = false;
        for (uint i=0; i<infos.size(); i++) {

            // get info
            const RpcInfo& info = infos[i];

            // do not handle finished paths
            if (info.path->finished)
                continue;

            // check if path accepts the message
            if ( !rpcHandled && info.path->accepts( info.vrpcId ) ) {
                info.path->handleResponse( findNodeResponse );
                rpcHandled = true;
            } else {
                info.path->handleTimeout( info.vrpcId );
            }

            // count finished and successful paths
            if (info.path->finished) {
                finishedPaths++;

                // count total number of hops
                accumulatedHops += info.path->hops;

                if (info.path->success)
                    successfulPaths++;
            }

        }
    }

    checkStop();
}


void BaseLookup::handleRpcTimeout( BaseCallMessage* msg, 
				   const NodeHandle& dest, int rpcId)
{
    // check flags
    if (finished || !running)
        return;

    // check if rpc info is available
    const NodeHandle& src = dest;
    if (rpcs.count(src)==0)
        return;
    RpcInfoVector& infos = rpcs[src];

    // iterate
    for (uint i=0; i < infos.size(); i++) {

        const RpcInfo& info = infos[i];

        // do not handle finished paths
        if (info.path->finished)
            continue;

        // delegate timeout
        info.path->handleTimeout( info.vrpcId );

        // count total number of hops
        accumulatedHops += info.path->hops;

        // count finished and successful paths
        if (info.path->finished) {
            finishedPaths++;
            if (info.path->success)
                successfulPaths++;
        }
    }

    checkStop();
}

void BaseLookup::sendRpc( const NodeHandle& handle, FindNodeCall* call,
                          BasePathLookup* listener, int rpcId )
{
    // check flags
    if (finished || !running)
        return;

    // create rpc info
    RpcInfo info;
    info.path = listener;
    info.vrpcId = rpcId;

    // send new message
    if ( rpcs.count(handle)==0 ) {
        RpcInfoVector newVector = RpcInfoVector();
        rpcs[handle] = newVector;

        if (overlay->measureNetwInitPhase ||
	    !overlay->underlayConfigurator->isInit()) {

	    overlay->numFindNodeSent++;
	    overlay->bytesFindNodeSent += call->byteLength();
	}

        newVector.nonce = overlay->sendRpcMessage( handle, call, this );
    }

    // register info
    rpcs[handle].push_back(info);
}

//----------------------------------------------------------------------------
//- AbstractLookup implementation --------------------------------------------
//----------------------------------------------------------------------------

void BaseLookup::lookup( const OverlayKey& key, uint numNeighbors,
			 int hopCountMax, LookupListener* listener )
{
    // check flags
    if (finished || running)
        return;

    // set params
    this->key = key;
    this->numNeighbors = numNeighbors;
    this->hopCountMax = hopCountMax;
    this->listener = listener;

    // start lookup
    start();
}

const NodeVector& BaseLookup::getResult() const
{
    // return neighbor vector
    return neighbors;
}

bool BaseLookup::isValid() const
{
    return success && finished;
}

uint BaseLookup::getAccumulatedHops() const
{
    return accumulatedHops;
}

BasePathLookup::BasePathLookup(BaseLookup* lookup)
{
    this->lookup = lookup;
    this->hops = 0;
    this->step = 0;
    this->pendingRpcs = 0;
    this->finished = false;
    this->success = false;
    this->nextHops = NodeVector( lookup->config.numNextHops, lookup );
}

BasePathLookup::~BasePathLookup()
{}

bool BasePathLookup::accepts( int rpcId )
{
    bool accept = ( rpcId == step ) && !finished;
    return accept;
}

void BasePathLookup::handleResponse( FindNodeResponse* msg )
{
    if (finished)
        return;

    // increase hops: FIXME don't count local hops
    if (lookup->overlay->getThisNode() != msg->getSrcNode())
        hops++;
    step++;

    // decrease pending rpcs
    pendingRpcs--;

    // mode: merge or replace
    if (!lookup->config.merge) {
        nextHops.clear();
    }

    // add new next hops
    for ( uint i=0; i < msg->getClosestNodesArraySize(); i++ ) {
        const NodeHandle& handle = msg->getClosestNodes(i);

        // add node handles to next hops and neighborhood
        add( handle );

        // check if node was found
        if ( handle.key == lookup->key && !lookup->config.secure ) {
            lookup->addNeighbor( handle );

            finished = true;
            success = true;
            return;
        } else
        if (lookup->numNeighbors != 0 && !lookup->config.secure && msg->getNeighbors() )
            lookup->addNeighbor( handle );

    }

    // check if neighborlookup is finished
    if ( msg->getNeighbors() && msg->getClosestNodesArraySize() != 0 &&
            lookup->numNeighbors != 0 && !lookup->config.secure ) {

        finished = true;
        success = true;
        return;
    }

    // extract find node extension object
    cMessage* findNodeExt = NULL;
    if (msg->hasObject("findNodeExt"))
        findNodeExt = (cMessage*)msg->removeObject("findNodeExt");

    // send next rpcs
    sendRpc( lookup->config.parallelRpcs, findNodeExt );

    // ...
    delete findNodeExt;
}


void BasePathLookup::handleTimeout( int rpcId )
{
    if (finished)
        return;

    EV << "BasePathLookup: Timeout of RPC " << rpcId << endl;

    // decrease pending rpcs
    pendingRpcs--;

    // last rpc? yes-> send next rpc
    if ( pendingRpcs == 0 )
        sendRpc( 1 );
}


void BasePathLookup::sendRpc( int num, cMessage* findNodeExt )
{
    // path finished? yes -> quit
    if (finished)
        return;

    // check for maximum hop count
    if (lookup->hopCountMax && (hops >= lookup->hopCountMax)) {
	EV << "BasePathLookup::sendRpc(): Max hop count exceeded - "
	   << "lookup failed!" << endl;
        finished = true;
        success = false;
	return;
    }

    // send rpc messages
    while ( num >= 0 && nextHops.size() != 0 ) {

        // get top node handle
        const NodeHandle& handle = nextHops.front();

        // check if node has already been visited? no ->
        if ( !lookup->getVisited( handle ) ) {

            // send rpc to node increase pending rpcs
            pendingRpcs++;
            num--;
            lookup->sendRpc( handle, createRpcMessage( findNodeExt ),
			     this, step );
        }

        // delete first element and continue
        nextHops.erase( nextHops.begin() );
    }

    // no rpc sent and no pending rpcs? -> failed
    if ( pendingRpcs == 0 ) {
        finished = true;
        success = false;
    }
}

FindNodeCall* BasePathLookup::createRpcMessage( cMessage* findNodeExt )
{
    // create default find node call message
    FindNodeCall* call = new FindNodeCall( "FindNodeCall" );
    call->setLookupKey( lookup->key );
    call->setLength( FINDNODECALL_L(call) );

    // duplicate extension object
    if ( findNodeExt != NULL ) {
        call->addObject( (cObject*)findNodeExt->dup() );
	call->addLength( findNodeExt->length() );
    }

    return call;
}

void BasePathLookup::add( const NodeHandle& handle )
{
    if ( lookup->config.merge )
        nextHops.add(handle);
    else
        nextHops.push_back(handle);
}

