//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseLookup.h
 * @author Sebastian Mies
 */

#ifndef __BASE_LOOKUP_H
#define __BASE_LOOKUP_H

// predefinitions
class LookupListener;
class AbstractLookup;
class BaseLookup;
class BasePathLookup;

/**
 * This class holds the lookup configuration.
 *
 * @author Sebastian Mies
 */
class BaseLookupConfiguration
{
public:
    int numNextHops;   //!< number of next hops in each step
    int parallelPaths; //!< number of parallel paths
    int parallelRpcs;  //!< number of nodes to ask in parallel
    bool secure;       //!< true, if all nodes should be identified with a ping
    bool merge;        //!< true, if parallel Rpc results should be merged
};

// includes
#include <BaseOverlay.h>
#include <RpcListener.h>

#include <OverlayKey.h>
#include <NodeHandle.h>
#include <NodeVector.h>
#include <Comparator.h>

#include <ext/hash_map>
#include <ext/hash_set>
#include <vector>

// used namespaces
using namespace std;
using namespace __gnu_cxx;


/**
 * This class declares an abstract lookup listener.
 * 
 * @author Sebastian Mies
 */
class LookupListener
{
public:
    virtual ~LookupListener();
    virtual void lookupFinished( AbstractLookup* lookup ) = 0;
};

/**
 * This class declares an abstract iterative lookup.
 *
 * @author Sebastian Mies
 */
class AbstractLookup
{
public:
    /**
     * Virtual destructor
     */
    virtual ~AbstractLookup();

    /**
     * Lookup a neighborhood or a key
     * 
     * @param key The key to lookup
     * @param numNeighbors Number of Neighbors to lookup
     * @param hopCountMax Maximum hop count
     * @param listener Listener to inform, when the lookup is done
     */
    virtual void lookup( const OverlayKey& key, uint numNeighbors = 0,
			 int hopCountMax = 0,
			 LookupListener* listener = NULL ) = 0;

    /**
     * Returns the result of the lookup
     * 
     * @return The result node vector.
     */
    virtual const NodeVector& getResult() const = 0;

    /**
     * Returns true, if the lookup was successful.
     * 
     * @return true, if the lookup was successful.
     */
    virtual bool isValid() const = 0;

    /**
     * Returns the total number of hops for all lookup paths.
     * 
     * @return The accumulated number of hops.
     */
    virtual uint getAccumulatedHops() const = 0;

};




/**
 * This class implements a basic greedy lookup strategy.
 * 
 * It uses the standard metric for greedy behaviour. If another
 * metric is needed, the distance function can be replaced by
 * overriding the distance method.
 *
 * @author Sebastian Mies
 */
class BaseLookup : public RpcListener,
		   public AbstractLookup,
		   public Comparator<OverlayKey>
{
    friend class BasePathLookup;
    friend class BaseOverlay;

    //-------------------------------------------------------------------------
    //- Enhanceable methods ---------------------------------------------------
    //-------------------------------------------------------------------------
protected:
    /**
     * This method creates a new path lookup. It may be overloaded
     * to enhance BasePathLookup with some new information/features.
     * 
     * @return The new path lookup
     */
    virtual BasePathLookup* createPathLookup();

    //-------------------------------------------------------------------------
    //- Base configuration and state ------------------------------------------
    //-------------------------------------------------------------------------
protected:
    OverlayKey key;                 //!< key to lookup
    BaseOverlay* overlay;           //!< ptr to overlay
    BaseLookupConfiguration config; //!< lookup configuration
    LookupListener* listener;       //!< lookup listener
    vector<BasePathLookup*> paths;  //!< parallel paths
    bool finished;                  //!< true, if lookup is finished
    bool success;                   //!< true, if lookup was successful
    bool running;                   //!< true, if lookup is running
    uint finishedPaths;             //!< number of finished paths
    uint successfulPaths;           //!< number of successful paths
    uint accumulatedHops;           //!< total number of hops (for all paths)

protected://virtual methods: comparator induced by distance in BaseOverlay
    int compare( const OverlayKey& lhs, const OverlayKey& rhs ) const;

    //-------------------------------------------------------------------------
    //- Neighbors and visited nodes management---------------------------------
    //-------------------------------------------------------------------------
protected://fields
    NodeVector neighbors;           //!< closest nodes
    NodeHandle::Set visited;        //!< nodes already visited
    int numNeighbors;               //!< number of neighbors
    int hopCountMax;               //!< maximum hop count

protected://methods
    bool addNeighbor( const NodeHandle& handle );
    void setVisited( const NodeHandle& handle, bool visited = true );
    bool getVisited( const NodeHandle& handle );

    //-------------------------------------------------------------------------
    //- Parallel RPC distribution ---------------------------------------------
    //-------------------------------------------------------------------------
protected://fields and classes: rpc distribution
    class RpcInfo
    {
    public:
        int vrpcId;
        BasePathLookup* path;
    };
class RpcInfoVector : public vector<RpcInfo>
    {
    public:
        uint32_t nonce;
    };
    typedef hash_map<NodeHandle, RpcInfoVector, NodeHandle::hashFcn>
	RpcInfoMap;
    RpcInfoMap rpcs;

protected://methods: rpcListener
    void handleRpcResponse(BaseResponseMessage* msg,int rpcId,simtime_t rtt );
    void handleRpcTimeout(BaseCallMessage* msg,const NodeHandle& dest,
			  int rpcId);

protected://methods: rpc distribution
    void sendRpc( const NodeHandle& handle, FindNodeCall* call,
                  BasePathLookup* listener, int rpcId );

    //-------------------------------------------------------------------------
    //- Construction & Destruction --------------------------------------------
    //-------------------------------------------------------------------------
protected://construction & destruction
    BaseLookup( BaseOverlay* overlay, const BaseLookupConfiguration& config );
    virtual ~BaseLookup();

    void start();
    void stop();
    void checkStop();

    //-------------------------------------------------------------------------
    //- AbstractLookup implementation -----------------------------------------
    //-------------------------------------------------------------------------
public://methods
    void lookup( const OverlayKey& key, uint numNeighbors = 0,
		 int hopCountMax = 0,
                 LookupListener* listener = NULL );

    const NodeVector& getResult() const;

    bool isValid() const;
    uint getAccumulatedHops() const;
};




/**
 * This class implements a path lookup.
 *
 * @author Sebastian Mies
 */
class BasePathLookup
{
    friend class BaseLookup;

protected://fields:
    BaseLookup* lookup;

protected://fields: state
    int  hops;
    int  step;
    int  pendingRpcs;
    bool finished;
    bool success;
    NodeVector nextHops;

protected://methods: rpc handling
    bool accepts( int rpcId );
    void handleResponse( FindNodeResponse* msg );
    void handleTimeout( int rpcId );

private:
    void sendRpc( int num, cMessage* FindNodeExt = NULL );

protected:
    BasePathLookup( BaseLookup* lookup );
    virtual ~BasePathLookup();

    /**
     * Creates a find node call message. This method can be
     * overridden to add some additional state information to the
     * FindNodeCall message.
     *
     * @param findNodeExt Pointer to a optional cMessage, that may
     *                    contain overlay specific data to be attached
     *                    to FindNode RPCs and BaseRouteMessages
     * 
     * @returns Pointer to a new FindNodeCall message.
     */
    virtual FindNodeCall* createRpcMessage(cMessage *findNodeExt = NULL);

    /**
     * Adds a NodeHandle to next hops
     */
    void add( const NodeHandle& handle );
};

#endif

