//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseOverlay.cc
 * @author Bernhard Heep (initial)
 * @author Sebastian Mies (rpcs)
 */

#include <UDPAppBase.h>
#include <UDPControlInfo_m.h>
#include <IPAddressResolver.h>

#include <CommonAPIMessages_m.h>
#include <BootstrapOracleAccess.h>
#include <UnderlayConfiguratorAccess.h>
#include <GlobalStatisticsAccess.h>


#include "BaseOverlay.h"
#include "RpcMacros.h"

//------------------------------------------------------------------------
//--- Initialization & finishing -----------------------------------------
//------------------------------------------------------------------------

BaseOverlay::~BaseOverlay()
{
    finishLookups();
    finishRpcs();
}

int BaseOverlay::numInitStages() const
{
    return MAX_STAGE_OVERLAY + 1;
}

void BaseOverlay::initialize(int stage)
{
    if(stage == MIN_STAGE_OVERLAY) {
        OverlayKey::setKeyLength(par("keyLength"));

        // fetch some parameters
        debugOutput                     = par("debugOutput");
        measureNetwInitPhase            = par("measureNetwInitPhase");
        localPort                       = par("localPort");
        hopCountMax                     = par("hopCountMax");
        drawOverlayTopology             = par("drawOverlayTopology");
        useBaseLookup                   = par("useBaseLookup");
        iterativeLookup                 = par("iterativeLookup");
        onlyCommonAPIMessages           = true;

        // set base lookup parameters
        baseLookupConfig.numNextHops    = par("lookupNumberNextHops");
        baseLookupConfig.parallelPaths  = par("lookupParallelPaths");
        baseLookupConfig.parallelRpcs   = par("lookupParallelRpcs");
        baseLookupConfig.secure         = par("lookupSecure");
        baseLookupConfig.merge          = par("lookupMerge");

        // statistics
        numSent = 0;
        bytesSent = 0;
        numSignalingSent = 0;
        bytesSignalingSent = 0;
        numReceived = 0;
        bytesReceived = 0;
        numSignalingReceived = 0;
        bytesSignalingReceived = 0;
        numForwarded = 0;
        bytesForwarded = 0;
        numSignalingForwarded = 0;
        bytesSignalingForwarded = 0;
        numDropped = 0;
        bytesDropped = 0;
	numFindNodeSent = 0;
	bytesFindNodeSent = 0;

        WATCH(numSent);
        WATCH(bytesSent);
        WATCH(numSignalingSent);
        WATCH(bytesSignalingSent);
        WATCH(numReceived);
        WATCH(bytesReceived);
        WATCH(numSignalingReceived);
        WATCH(bytesSignalingReceived);
        WATCH(numForwarded);
        WATCH(bytesForwarded);
        WATCH(numSignalingForwarded);
        WATCH(bytesSignalingForwarded);
        WATCH(numDropped);
        WATCH(bytesDropped);
	WATCH(numFindNodeSent);
	WATCH(bytesFindNodeSent);

        // set up UDP
        bindToPort(localPort);

        // find friend modules
        bootstrapOracle = BootstrapOracleAccess().get();
        underlayConfigurator = UnderlayConfiguratorAccess().get();
        globalStatistics = GlobalStatisticsAccess().get();
        notificationBoard = NotificationBoardAccess().get();

        thisTerminal = parentModule()->parentModule();

        // set up local nodehandle
        thisNode.moduleId = parentModule()->id();
        thisNode.ip = IPAddressResolver().
                      addressOf(parentModule()->parentModule()).get4();
        thisNode.port = localPort;
        thisNode.key = OverlayKey::UNSPECIFIED_KEY;

        // set up arrow-gates
        thisOutGateArray = thisTerminal->gate("overlayNeighborArrowOut");
        thisInGateArray = thisTerminal->gate("overlayNeighborArrowIn");
        thisTerminal->setGateSize("overlayNeighborArrowOut" ,1);
        thisTerminal->setGateSize("overlayNeighborArrowIn" ,1);

        // subscribe to the notification board
        notificationBoard->subscribe( this,NF_HOSTPOSITION_UPDATED);

        // init rpcs
        initRpcs();
        initLookups();
    }

    if(stage >= MIN_STAGE_OVERLAY && stage <= MAX_STAGE_OVERLAY)
        initializeOverlay(stage);
}

void BaseOverlay::initializeOverlay(int stage)
{
    // ...
}

void BaseOverlay::finish()
{
    finishOverlay();
    finishLookups();
    finishRpcs();

    recordScalar("BaseOverlay: Sent User Messages", numSent);
    recordScalar("BaseOverlay: Sent User Bytes", bytesSent);
    recordScalar("BaseOverlay: Sent Signaling Messages", numSignalingSent);
    recordScalar("BaseOverlay: Sent Signaling Bytes", bytesSignalingSent);
    recordScalar("BaseOverlay: Sent Total Messages", numSent + numSignalingSent);
    recordScalar("BaseOverlay: Sent Total Bytes", bytesSent + bytesSignalingSent);
    recordScalar("BaseOverlay: Sent FindNode Messages", numFindNodeSent);
    recordScalar("BaseOverlay: Sent FindNode Bytes", bytesFindNodeSent);

    recordScalar("BaseOverlay: Received User Messages", numReceived);
    recordScalar("BaseOverlay: Received User Bytes", bytesReceived);
    recordScalar("BaseOverlay: Received Signaling Messages", numSignalingReceived);
    recordScalar("BaseOverlay: Received Signaling Bytes", bytesSignalingReceived);
    recordScalar("BaseOverlay: Received Total Messages", numReceived
                 + numSignalingReceived);
    recordScalar("BaseOverlay: Received Total Bytes", bytesReceived
                 + bytesSignalingReceived);
    recordScalar("BaseOverlay: Forwarded User Messages", numForwarded);
    recordScalar("BaseOverlay: Forwarded User Bytes", bytesForwarded);
    recordScalar("BaseOverlay: Forwarded Signaling Messages", numSignalingForwarded);
    recordScalar("BaseOverlay: Forwarded Signaling Bytes", bytesSignalingForwarded);
    recordScalar("BaseOverlay: Forwarded Total Messages", numForwarded
                 + numSignalingForwarded);
    recordScalar("BaseOverlay: Forwarded Total Bytes", bytesForwarded
                 + bytesSignalingForwarded);

    recordScalar("BaseOverlay: Dropped Messages", numDropped);
    recordScalar("BaseOverlay: Dropped Bytes", bytesDropped);

}

void BaseOverlay::finishOverlay()
{
    // ...
}

//------------------------------------------------------------------------
//--- General Overlay Parameters (getter and setters) --------------------
//------------------------------------------------------------------------

void BaseOverlay::setMalicious( bool malicious )
{
    this->malicious = malicious;
}

bool BaseOverlay::isMalicious()
{
    return malicious;
}

NodeHandle& BaseOverlay::getThisNode()
{
    return thisNode;
}


//------------------------------------------------------------------------
//--- Overlay Common API: Key-based Routing ------------------------------
//------------------------------------------------------------------------

void BaseOverlay::route( const OverlayKey& key, cMessage* msg,
                         const NodeHandle& hint )
{}

bool BaseOverlay::isResponsible(const OverlayKey& key)
{
    opp_error( "isResponsible: Not implemented!" );
    return false;
}


void BaseOverlay::callDeliver(BaseOverlayMessage* msg)
{
    KBRdeliver* deliverMsg = new KBRdeliver();

    OverlayCtrlInfo* overlayCtrlInfo =
        check_and_cast<OverlayCtrlInfo*>
        (msg->removeControlInfo());

    deliverMsg->setControlInfo(overlayCtrlInfo);

    deliverMsg->encapsulate(msg->decapsulate());

    send(deliverMsg, "to_app");

    delete msg;
}

void BaseOverlay::callForward(OverlayKey& key, BaseRouteMessage* msg,
                              NodeHandle* hint)
{
    KBRforward* forwardMsg = new KBRforward();
    forwardMsg->encapsulate(msg->decapsulate());

    OverlayCtrlInfo* overlayCtrlInfo =
        new OverlayCtrlInfo();
    overlayCtrlInfo->setThisNode(thisNode);
    overlayCtrlInfo->setHopCount(msg->getHopCount());
    overlayCtrlInfo->setDestKey(msg->getDestKey());
    overlayCtrlInfo->setSrcNode(msg->getSrcNode());

    forwardMsg->setControlInfo(overlayCtrlInfo);

    send(forwardMsg, "to_app");

    delete msg;
}


//------------------------------------------------------------------------
//--- Message Handlers ---------------------------------------------------
//------------------------------------------------------------------------

//private
void BaseOverlay::handleMessage(cMessage* msg)
{
    // process self-messages
    if (msg->isSelfMessage()) {
        // process rpc self-messages
        BaseRpcMessage* rpcMessage = dynamic_cast<BaseRpcMessage*>(msg);
        if (rpcMessage!=NULL) {
            internalHandleRpcMessage(rpcMessage);
            return;
        }
        // process all other self-messages
        handleTimerEvent(msg);
    }

    // process messages from UDP
    else if (msg->arrivedOn("from_udp")) {
        UDPControlInfo* udpControlInfo =
            check_and_cast<UDPControlInfo*>(msg->removeControlInfo());
        OverlayCtrlInfo* overlayCtrlInfo = new OverlayCtrlInfo;
        overlayCtrlInfo->setLastHopAddr(udpControlInfo->srcAddr());
        overlayCtrlInfo->setLastHopPort(udpControlInfo->srcPort());
        msg->setControlInfo(overlayCtrlInfo);
        delete udpControlInfo;

        BaseOverlayMessage* baseOverlayMsg =
            dynamic_cast<BaseOverlayMessage*>(msg);

        if (baseOverlayMsg == NULL) {
            delete msg;
            return;
        }

        // records stats if message is not a UDP "self message"
        if (overlayCtrlInfo->getLastHopAddr() != thisNode.ip) {
            if (baseOverlayMsg->getSignaling() == false)
                RECORD_STATS(numReceived++; bytesReceived +=
                                 baseOverlayMsg->byteLength());
            else
                RECORD_STATS(numSignalingReceived++;bytesSignalingReceived
                             += baseOverlayMsg->byteLength());
        }

        handleBaseOverlayMessage(baseOverlayMsg);
    }

    // process CommonAPIMessages from App
    else if (dynamic_cast<CommonAPIMessage*>(msg) != NULL) {
        OverlayCtrlInfo* overlayCtrlInfo =
            check_and_cast<OverlayCtrlInfo*>
            (msg->removeControlInfo());

        if (dynamic_cast<KBRroute*>(msg) != NULL)
            internalRoute(overlayCtrlInfo->getDestKey(), msg->decapsulate(),
			  overlayCtrlInfo->getHint());

        delete msg;
        delete overlayCtrlInfo;
    }

    // process other messages from App
    else if (msg->arrivedOn("from_app") && onlyCommonAPIMessages == false)
        handleAppMessage(msg);

    else {
        RECORD_STATS(numDropped++; bytesDropped += msg->byteLength());
        delete msg;
    }
}

void BaseOverlay::handleBaseOverlayMessage(BaseOverlayMessage* msg)
{
    switch(msg->getType()) {
    case OVERLAYSIGNALING:
        handleUDPMessage(msg);
        return;

    case RPC: {
            // process rpc-messages
            BaseRpcMessage* rpcMessage =
                check_and_cast<BaseRpcMessage*>(msg);
            internalHandleRpcMessage(rpcMessage);
            return;
        }

    case APPDATA: {
            BaseAppDataMessage* baseAppDataMsg =
                check_and_cast<BaseAppDataMessage*>(msg);
            callDeliver(baseAppDataMsg);
            return;
        }

    case OVERLAYROUTE: {
            BaseRouteMessage* baseRouteMsg =
                check_and_cast<BaseRouteMessage*>(msg);

            if (baseRouteMsg->getDestKey().isUnspecified())
                opp_error("BaseOverlay: No destination key in ROUTE msg");

            if (baseRouteMsg->getFinalHop()) {

                // check if this node is really resposible for this key
		if (!isResponsible(baseRouteMsg->getDestKey())) {
		    EV << "BaseOverlay: Received BaseRouteMessage with "
		       << "finalHopFlag set, but this node "
		       << "is not responsible - dropping message!" << endl;
		    RECORD_STATS(numDropped++;
				 bytesDropped += baseRouteMsg->byteLength());
		    delete msg;
		    return;
		}

                OverlayCtrlInfo* overlayCtrlInfo =
                    check_and_cast<OverlayCtrlInfo*>(msg->removeControlInfo());

                overlayCtrlInfo->setThisNode(thisNode);
                overlayCtrlInfo->setHopCount(baseRouteMsg->getHopCount());
                overlayCtrlInfo->setDestKey(baseRouteMsg->getDestKey());
                overlayCtrlInfo->setSrcNode(baseRouteMsg->getSrcNode());

                BaseOverlayMessage* tmpMsg
                = check_and_cast<BaseOverlayMessage*>
                  (baseRouteMsg->decapsulate());
                tmpMsg->setControlInfo(overlayCtrlInfo);

                handleBaseOverlayMessage(tmpMsg);
                delete msg;
                return;
            } else {
                // forward msg if this node is not responsible for the key
                sendToKey(baseRouteMsg->getDestKey(), baseRouteMsg, 1);
                return;
            }
            break;
        }

    default:
        EV << "BaseOverlay::handleMessage(): received unknown message "
        << "from UDP of type " << msg->name() << endl;
        break;
    }
}


//virtual protected
void BaseOverlay::handleTimerEvent(cMessage* msg)
{
    // timer...
}

//virtual protected
void BaseOverlay::handleAppMessage(cMessage* msg)
{
    delete msg;
}

//virtual protected
void BaseOverlay::recordOverlaySentStats(BaseOverlayMessage* msg)
{
    // collect statistics ...
}


//------------------------------------------------------------------------
//--- Overlay Topology Drawing -------------------------------------------
//------------------------------------------------------------------------

void BaseOverlay::setReadyIcon(bool ready)
{
    if (ev.isGUI()) {
        if (ready) {
            thisTerminal->displayString().setTagArg("i2", 1, "");
            displayString().setTagArg("i", 1, "");
        } else {
            thisTerminal->displayString().setTagArg("i2", 1, "red");
            displayString().setTagArg("i", 1, "red");
        }
    }
}

void BaseOverlay::showOverlayNeighborArrow(const NodeHandle& neighbor,
        bool flush, char* displayString)
{
    if (!ev.isGUI() || !drawOverlayTopology)
        return;

    char red[] = "o=red,1";
    if (displayString == NULL)
        displayString = red;

    cModule* neighborTerminal;
    cGate* neighborInGateArray;

    // flush
    if (flush) {
        for (int l = 0; l < thisOutGateArray->size(); l++) {
            cGate* tempGate
            = thisTerminal->gate("overlayNeighborArrowOut", l)->toGate();
            thisTerminal->gate("overlayNeighborArrowOut", l)->disconnect();
            if (tempGate != NULL)
                compactGateArray(tempGate->ownerModule(), IN);
        }
        thisTerminal->setGateSize("overlayNeighborArrowOut" ,0);
    }

    if (simulation.module(neighbor.moduleId) != NULL) {
        neighborTerminal
        = simulation.module(neighbor.moduleId)->parentModule();
        neighborInGateArray = simulation.module(neighbor.moduleId)->
                              parentModule()->gate("overlayNeighborArrowIn");
    } else
        return;

    if (thisTerminal == neighborTerminal)
        return;

    //do not draw double
    for (int i = 0; i < thisOutGateArray->size(); i++)
        if (thisTerminal->gate("overlayNeighborArrowOut", i)->toGate() != NULL
                && neighborTerminal
                == thisTerminal->gate("overlayNeighborArrowOut", i)->
                toGate()->ownerModule())
            return;

    // IN
    int i = 0;
    if (neighborInGateArray->size() == 0)
        neighborTerminal->setGateSize("overlayNeighborArrowIn", 1);
    else {
        for (i = 0; i < neighborInGateArray->size() - 1; i++) {
            if (!(neighborTerminal->gate("overlayNeighborArrowIn", i)
                    ->isConnectedOutside()))
                break;
        }
        if (neighborTerminal->gate("overlayNeighborArrowIn", i)
                ->isConnectedOutside()) {
            neighborTerminal->setGateSize("overlayNeighborArrowIn", i + 2);
            i++;
        }
    }

    // OUT
    int j = 0;
    if (thisOutGateArray->size() == 0)
        thisTerminal->setGateSize("overlayNeighborArrowOut", 1);
    else {
        for (j = 0; j < (thisOutGateArray->size() - 1); j++) {
            if (!(thisTerminal->gate("overlayNeighborArrowOut", j)
                    ->isConnectedOutside()))
                break;
        }
        if (thisTerminal->gate("overlayNeighborArrowOut", j)
                ->isConnectedOutside()) {
            thisTerminal->setGateSize("overlayNeighborArrowOut", j + 2);
            j++;
        }
    }

    thisTerminal->gate("overlayNeighborArrowOut", j)->
    connectTo(neighborTerminal->gate("overlayNeighborArrowIn", i));

    thisTerminal->gate("overlayNeighborArrowOut", j)->
    setDisplayString(displayString);
}

void BaseOverlay::deleteOverlayNeighborArrow(const NodeHandle& neighbor)
{
    if (!ev.isGUI() || !drawOverlayTopology)
        return;

    //does neighbor module exist anymore?
    if (simulation.module(neighbor.moduleId) == NULL)
        return;

    cModule* neighborTerminal
    = simulation.module(neighbor.moduleId)->parentModule();

    //find gate
    bool compactOut = false;
    bool compactIn = false;
    for (int i = 0; i < thisOutGateArray->size(); i++) {
        // NULL-Gate?
        if (thisTerminal->gate("overlayNeighborArrowOut", i)->toGate() == NULL) {
            compactOut = true;
            continue;
        }

        if (thisTerminal->gate("overlayNeighborArrowOut", i)->toGate()->ownerModule()->id() == neighborTerminal->id()) {
            thisTerminal->gate("overlayNeighborArrowOut", i)->disconnect();
            compactOut = true;
            compactIn = true;
        }
    }

    //compact OUT-array
    if (compactOut)
        compactGateArray(thisTerminal, OUT);
    //compact IN-array
    if (compactIn)
        compactGateArray(neighborTerminal, IN);
}

void BaseOverlay::compactGateArray(cModule* terminal, enum Direction dir)
{
    cGate* gateArray = (dir == OUT ? terminal->gate("overlayNeighborArrowOut")
                        : terminal->gate("overlayNeighborArrowIn"));
    const char* gateName = (dir == OUT ? "overlayNeighborArrowOut"
                            : "overlayNeighborArrowIn");

    for (int j = 0; j < gateArray->size() - 1; j++) {
        if (terminal->gate(gateName, j)->isConnectedOutside())
            continue;

        cGate* tempGate = NULL;
        int k = 1;
        while ((tempGate == NULL) && ((j + k) != gateArray->size())) {
            tempGate = (dir == OUT ? terminal->gate(gateName, j + k)->toGate()
                        : terminal->gate(gateName, j + k)->fromGate());
            k++;
        }

        if (tempGate == NULL)
            break;

        if (dir == OUT) {
            terminal->gate(gateName, j + k - 1)->disconnect();
            terminal->gate(gateName, j)->connectTo(tempGate);
        } else {
            tempGate->disconnect();
            tempGate->connectTo(terminal->gate(gateName, j));
        }
    }

    int nullGates = 0;
    for (int j = 0; j < gateArray->size(); j++)
        if (!terminal->gate(gateName, j)->isConnectedOutside())
            nullGates++;

    terminal->setGateSize(gateName, gateArray->size() - nullGates);
}

//------------------------------------------------------------------------
//--- Messages -----------------------------------------------------------
//------------------------------------------------------------------------

void BaseOverlay::sendMessageToUDP(const NodeHandle& dest,
                                   BaseOverlayMessage* msg)
{
    // if there's still a control info attached to the message, remove it
    cPolymorphic* ctrlInfo = msg->removeControlInfo();
    if (ctrlInfo != NULL)
        delete ctrlInfo;

    // debug message
    if (debugOutput) {
        EV << "BaseOverlay: Node " << thisNode.ip << ": sends "
        << msg->name() << " to " << dest.ip << "." << endl;
    }

    msg->setKind(UDP_C_DATA);
    UDPControlInfo* udpControlInfo = new UDPControlInfo();
    udpControlInfo->setSrcAddr(thisNode.ip);
    udpControlInfo->setSrcPort(thisNode.port);
    udpControlInfo->setDestAddr(dest.ip);
    udpControlInfo->setDestPort(dest.port);
    msg->setControlInfo(udpControlInfo);

    send(msg, "to_udp");

    if (dest != thisNode) {
        // record statistics, if message is not local
        if (msg->getSignaling() == false) {
            RECORD_STATS(numSent++; bytesSent += msg->byteLength());
        } else {
            RECORD_STATS(numSignalingSent++; bytesSignalingSent +=
                             msg->byteLength());
        }
        recordOverlaySentStats(msg);
	globalStatistics->sentTotalBytes+=msg->byteLength();

    }
}


//------------------------------------------------------------------------
//--- RPC Handling -------------------------------------------------------
//------------------------------------------------------------------------

//private
void BaseOverlay::initRpcs()
{
    rpcsPending = 0;
    rpcStates.clear();

    defaultRpcListener = new RpcListener();
}

//private
void BaseOverlay::finishRpcs()
{
    // stop all rpcs
    for (RpcStates::iterator i = rpcStates.begin();
            i != rpcStates.end(); i++) {
        cancelAndDelete(i->second.callMsg);
        cancelAndDelete(i->second.timeoutMsg);
    }
    rpcStates.clear();

    // delete default rpc listener
    if (defaultRpcListener!=NULL) {
        delete defaultRpcListener;
        defaultRpcListener = NULL;
    }
}

//public
uint32_t BaseOverlay::sendRpcMessage( const NodeHandle& dest,
                                      BaseCallMessage* msg,
                                      RpcListener* rpcListener,
                                      const OverlayKey& destKey,
                                      int rpcId, simtime_t timeout,
                                      int retries )
{

    // create nonce, timeout and set default parameters
    uint nonce;
    do {
        nonce = intuniform( 0, 2147483647);
    } while (rpcStates.count(nonce) > 0);


    if (timeout == -1)
        timeout = destKey.isUnspecified() ? 1.0 : 5.0;
    if (rpcListener == NULL)
        rpcListener = defaultRpcListener;

    // create state
    RpcState state;
    state.id = rpcId;
    state.timeSent = simulation.simTime();
    state.dest = dest;
    state.destKey = destKey;
    state.listener = rpcListener;
    state.timeoutMsg = new RpcTimeoutMessage();
    state.timeoutMsg->setNonce(nonce);
    state.retries = retries;

    if (rpcStates.count(nonce) > 0)
        error("RPC nonce collision");

    // set message parameters
    msg->setNonce( nonce );
    msg->setSrcNode( thisNode );
    msg->setType(RPC);

    // save copy of call message in RpcState
    state.callMsg = dynamic_cast<BaseCallMessage*>(msg->dup());

    // register state
    rpcStates[nonce] = state;

    // schedule timeout message
    if (timeout!=0)
        scheduleAt( simulation.simTime() + timeout, state.timeoutMsg);

    if (destKey.isUnspecified()) {
        sendMessageToUDP( dest, msg );
    } else {
        // send message to key
        sendToKey( destKey, msg, 1, dest );
    }

    return nonce;
}

//public
void BaseOverlay::cancelRpcMessage( uint32_t nonce )
{
    if (rpcStates.count(nonce)==0)
        return;
    RpcState state = rpcStates[nonce];
    rpcStates.erase(nonce);
    cancelAndDelete(state.callMsg);
    cancelAndDelete(state.timeoutMsg);
}

//private
void BaseOverlay::internalHandleRpcMessage( BaseRpcMessage* msg )
{
    // check if this is a rpc call message
    BaseCallMessage* rpCall = dynamic_cast<BaseCallMessage*>(msg);
    if (rpCall != NULL) {
        if (!internalHandleRpc(rpCall))
            handleRpc(rpCall);
        return;
    }

    // get nonce
    int nonce = msg->getNonce();

    // nonce known? no -> delete message and return
    if (rpcStates.count(nonce)==0) {
        EV << "RPC: Nonce Unknown!" << endl;
        delete msg;
        return;
    }

    // get state and remove from map
    RpcState state = rpcStates[nonce];
    rpcStates.erase(nonce);

    // is timeout message?
    if ( msg->isSelfMessage() ) { // yes-> inform listener

        // retry?
        state.retries--;
        simtime_t timeout = simulation.simTime() - state.timeSent;
        if (state.retries>=0) {
            if (state.destKey.isUnspecified()) {
                sendMessageToUDP( state.dest,
                                  dynamic_cast<BaseCallMessage*>
                                  (state.callMsg->dup()));
            } else {
                // send message to key
                sendToKey( state.destKey,
                           dynamic_cast<BaseCallMessage*>
                           (state.callMsg->dup()), 1, state.dest );
            }
            if (timeout!=0)
                scheduleAt( simulation.simTime() + timeout, msg );
            rpcStates[nonce] = state;
            return;
        }

        // inform listener
        if ( state.listener != NULL )
            state.listener->handleRpcTimeout( state.callMsg, state.dest,
                                              state.id );

        // inform overlay
        handleRpcTimeout( state.callMsg, state.dest, state.id );

    } else { // no-> handle rpc response

        // get parameters
        simtime_t rtt = simulation.simTime() - state.timeSent;
        BaseResponseMessage* response
        = dynamic_cast<BaseResponseMessage*>(msg);

        // inform listener
        if ( state.listener != NULL )
            state.listener->handleRpcResponse( response, state.id, rtt );

        // inform overlay
        handleRpcResponse( response, state.id, rtt );

        // delete response
        delete response;
    }

    // delete messages
    cancelAndDelete(state.callMsg);
    cancelAndDelete(state.timeoutMsg);

    // clean up pointers
    state.callMsg = NULL;
    state.timeoutMsg = NULL;
}

//private
bool BaseOverlay::internalHandleRpc( BaseCallMessage* msg )
{
    // call rpc stubs
    RPC_SWITCH_START( msg );
    RPC_DELEGATE( FindNode, findNodeRpc );
    RPC_SWITCH_END( );

    // check if rpc has been handled
    IF_RPC_HANDLED return true;
    else
        return false;
}

//virtual protected
void BaseOverlay::handleRpc( BaseCallMessage* msg )
{
    opp_error( "handleRpc: Not implemented!" );
    delete msg;
}

//protected
void BaseOverlay::sendRpcResponse( BaseCallMessage* call,
                                   BaseResponseMessage* response )
{
    if (call==NULL || response==NULL) {
        if ( call!=NULL )
            delete call;
        return;
    }
    response->setSrcNode( thisNode );
    response->setType( RPC );
    response->setNonce( call->getNonce() );
    sendMessageToUDP( call->getSrcNode(), response );
    delete call;
}


//------------------------------------------------------------------------
//--- Basic Routing ------------------------------------------------------
//------------------------------------------------------------------------

static int pendingLookups = 0;

void BaseOverlay::initLookups()
{
    lookups = LookupSet();
}

void BaseOverlay::finishLookups()
{
    while (lookups.size()!=0)
        delete *lookups.begin();
    lookups.clear();
}

class SendToKeyListener : public LookupListener
{
private:
    BaseOverlay* overlay;
    BaseRouteMessage* msg;
public:
    SendToKeyListener( BaseOverlay* overlay, BaseRouteMessage* msg )
    {
        this->overlay = overlay;
        this->msg = msg;
        pendingLookups++;
    }

    ~SendToKeyListener()
    {
        this->overlay = NULL;
        this->msg = NULL;
    }

    virtual void lookupFinished(AbstractLookup *lookup)
    {
        pendingLookups--;
        if (lookup->isValid()) {
            if (lookup->getResult().size()==0) {
                cout << "[ERROR] SendToKeyListener: Valid result, "
                "but empty array." << endl;
            } else {
                msg->setHopCount(lookup->getAccumulatedHops());
                msg->setFinalHop(true);

                for (uint i=0; i<lookup->getResult().size(); i++) {
                    overlay->sendMessageToUDP(lookup->getResult()[i],
                                              (BaseOverlayMessage*)msg->dup());
                }
            }
        }
        delete msg;
        delete this;
    }
};

void BaseOverlay::internalRoute(const OverlayKey& key, cMessage* msg,
                                const NodeHandle& hint)
{
    // check if base lookup should be used
    if (!useBaseLookup) {
        route(key,msg,hint);
        return;
    }

    if(key.isUnspecified())
        error("route(): key unspecified!");

    // create base route message
    BaseAppDataMessage* baseAppDataMsg =
        new BaseAppDataMessage("BaseAppDataMessage");
    baseAppDataMsg->setType(APPDATA);
    baseAppDataMsg->setLength(BASEAPPDATA_L(baseAppDataMsg));
    baseAppDataMsg->setSignaling(false);
    baseAppDataMsg->setKind(1);
    baseAppDataMsg->encapsulate(msg);

    // debug output
    if (debugOutput) {
        EV << "BaseOverlay: Node " << thisNode.ip << " received message from "
        "application." << endl;
    }

    sendToKey( key, baseAppDataMsg, 1, hint );
}

void BaseOverlay::sendToKey(const OverlayKey& key, BaseOverlayMessage* msg,
                            uint numNeighbors, const NodeHandle& nextHop)
{
    BaseRouteMessage* routeMsg = NULL;


    if (key.isUnspecified())
        error("BaseOverlay::sendToKey(): unspecified destination key!");

    if (msg->getType() != OVERLAYROUTE) {
        routeMsg = new BaseRouteMessage("BaseRouteMessage");
        routeMsg->setType(OVERLAYROUTE);
        routeMsg->setDestKey(key);
        routeMsg->setSrcNode(thisNode);
        routeMsg->setSignaling(msg->getSignaling());
        routeMsg->setKind(routeMsg->getSignaling() ? 0 : 1);
        // copy the name of the inner message
        routeMsg->setName(msg->name());
        routeMsg->setLength(BASEROUTE_L(routeMsg));
        routeMsg->encapsulate(msg);
    } else {
        routeMsg = check_and_cast<BaseRouteMessage*>(msg);
    }

    if (!nextHop.isUnspecified()) {
        // send msg to nextHop if specified (used for join rpcs)
        sendMessageToUDP(nextHop, routeMsg);
        return;
    }

    if (iterativeLookup) {
        // create lookup and sent to key
        AbstractLookup* lookup = createLookup();
        lookup->lookup( routeMsg->getDestKey(), numNeighbors, hopCountMax,
                        new SendToKeyListener( this, routeMsg ) );
    } else  {
        // if this node knows the destination node, set the finalHop flag
        if(isResponsible(routeMsg->getDestKey())) {
            routeMsg->setFinalHop(true);
        }

        NodeVector* nextHops = findNode(routeMsg->getDestKey(), routeMsg);

        if (nextHops->size() == 0) {

            EV <<   "BaseOverlay::sendToKey(): findNode() returned NULL "
            "- dropping message" << endl;
            // statistics
            RECORD_STATS(numDropped++; bytesDropped += routeMsg->byteLength());
            delete routeMsg;
        } else {
            // delete message if the hop count maximum is exceeded
            if (routeMsg->getHopCount() >= hopCountMax) {
                if (debugOutput) {
                    EV << "BaseOverlay: Node " << thisNode.ip << " discards "
                    << routeMsg->name() << " from "
                    << routeMsg->getSrcNode().ip
                    << ". The hop count maximum has been exceeded. ("
                    << routeMsg->getHopCount() << ">="
                    << hopCountMax << ")" << endl;
                    //cout << "HopCount exceeded!" << endl;
                }

                // statistics
                RECORD_STATS(numDropped++;
                             bytesDropped += routeMsg->byteLength());
                delete routeMsg;
                return;
            }

            // forward msg if this node is not responsible for the key

            if ((*nextHops)[0] != thisNode) {
                // increase hop count for non-local messages
                routeMsg->setHopCount(routeMsg->getHopCount() + 1);

                OverlayCtrlInfo* overlayCtrlInfo =
                    dynamic_cast<OverlayCtrlInfo*>(msg->removeControlInfo());

                // records statistics, if we forward this message
                if ((overlayCtrlInfo != NULL) &&
                        (overlayCtrlInfo->getLastHopAddr() != thisNode.ip)) {
                    if (routeMsg->getSignaling() == false) {
                        RECORD_STATS(numForwarded++; bytesForwarded +=
                                         routeMsg->byteLength());
                    } else {
                        RECORD_STATS(numSignalingForwarded++;
                                     bytesSignalingForwarded +=
                                         routeMsg->byteLength());
                    }
                }
                if (overlayCtrlInfo != NULL)
                    delete overlayCtrlInfo;
            }
            sendMessageToUDP((*nextHops)[0], routeMsg);
        }
        delete nextHops;
    }
}

//protected: create a iterative lookup class
AbstractLookup* BaseOverlay::createLookup()
{
    AbstractLookup* newLookup = new BaseLookup( this, baseLookupConfig );
    lookups.insert( newLookup );
    return newLookup;
}

void BaseOverlay::removeLookup( AbstractLookup* lookup )
{
    lookups.erase(lookup);
}

//virtual public
OverlayKey BaseOverlay::distance( const OverlayKey& x,
                                  const OverlayKey& y ) const
{
    return x > y ? x-y : y-x;
}

//protected: find closest nodes
NodeVector* BaseOverlay::findNode(const OverlayKey& key,
                                  BaseOverlayMessage* msg)
{
    opp_error( "findNode: Not implemented!" );
    return NULL;
}


PingResponse* BaseOverlay::ping( PingCall* call )
{
    return new PingResponse();
}

//private: rpc stub
void BaseOverlay::findNodeRpc( FindNodeCall* call )
{
    FindNodeResponse* findNodeResponse =
        new FindNodeResponse("FindNodeResponse");

    NodeVector* nextHops = findNode(call->getLookupKey(), call);

    findNodeResponse->setClosestNodesArraySize(nextHops->size());

    for (uint i=0; i < nextHops->size(); i++) {
        findNodeResponse->setClosestNodes(i, (*nextHops)[i]);
    }

    if (isResponsible(call->getLookupKey()))
        findNodeResponse->setNeighbors(true);

    findNodeResponse->setLength(FINDNODERESPONSE_L(findNodeResponse));
 
    if (call->hasObject("findNodeExt")) {
	cMessage* findNodeExt = (cMessage*)call->removeObject("findNodeExt");
	findNodeResponse->addObject(findNodeExt);
	findNodeResponse->addLength(findNodeExt->length());
    }

    RECORD_STATS(numFindNodeSent++; bytesFindNodeSent +=
		 findNodeResponse->byteLength());

    delete nextHops;

    sendRpcResponse( call, findNodeResponse );
}

//private: rpc stub
void BaseOverlay::pingRpc( PingCall* call )
{
    sendRpcResponse( call, ping(call) );
}

