// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file BaseOverlay.h
 * @author Bernhard Heep
 * @author Sebastian Mies
 */

#ifndef __BASEOVERLAY_H_
#define __BASEOVERLAY_H_

#include <omnetpp.h>

// global modules
#include <BootstrapOracle.h>
#include <GlobalStatistics.h>
#include <UnderlayConfigurator.h>
#include <InitStages.h>
#include <NotificationBoard.h>
#include <UDPAppBase.h>

// common structures
#include <NodeHandle.h>
#include <OverlayKey.h>
#include <BaseOverlayMessage_m.h>

// rpc includes
#include <BaseOverlayMessage_m.h>
#include <RpcListener.h>
#include <ext/hash_map>

// base lookup
#include <BaseLookup.h>

using namespace __gnu_cxx;

/**
 * Macro used for recording statistics considering measureNetwIn
 * parameter. The do-while-loop is needed for compatability in
 * outer if-else-structures.
 */
#define RECORD_STATS(x) \
    do { \
        if (measureNetwInitPhase||!underlayConfigurator->isInit()){ x; } \
    } while(false)


/**
 * Base class for overlays
 *
 * Base class for overlay modules, with KBR-API, statistics and
 * pointers to the BootstrapOracle and the UnderlayConfigurator.
 * A minimal overlay has to implement numInitStages(), route(),
 * handleUDPMessage() and receiveChangeNotification(). Derived
 * classes must use BaseOverlayMessage as base class for own
 * message types.
 *
 * @author Bernhard Heep (initial)
 * @author Sebastian Mies (rpc, lookup)
 */
class BaseOverlay : public UDPAppBase,
            public INotifiable,
            public RpcListener
{

    friend class BaseLookup;

    //------------------------------------------------------------------------
    //--- Construction / Destruction -----------------------------------------
    //------------------------------------------------------------------------

public:

    /** Virtual destructor */
    virtual ~BaseOverlay();

    //------------------------------------------------------------------------
    //--- Statistics ---------------------------------------------------------
    //------------------------------------------------------------------------

private://fields: statistics

    int numSent;   //!< number of sent packets (incl.\ forwarded packets)
    int bytesSent; //!< number of sent bytes (incl.\ forwarded bytes)
    int numSignalingSent;   //!< number of sent signaling packets (incl.\ forwarded packets)
    int bytesSignalingSent; //!< number of sent signaling bytes (incl.\ forwarded bytes)

    int numReceived;   //!< number of received packets (incl.\ packets to be forwarded )
    int bytesReceived; //!< number of received bytes (incl.\ bytes to be forwarded)
    int numSignalingReceived;   //!< number of received signaling packets (incl.\ packets to be forwarded)
    int bytesSignalingReceived; //!< number of received signaling bytes (incl.\ bytes to be forwarded)

protected:

    int numForwarded;   //!< number of forwarded packets
    int bytesForwarded; //!< number of forwarded bytes at out-gate
    int numSignalingForwarded;   //!< number of forwarded signaling packets
    int bytesSignalingForwarded; //!< number of forwarded signaling bytes at out-gate
    int numFindNodeSent; /**< */
    int bytesFindNodeSent; /**< */


    //------------------------------------------------------------------------
    //--- Common Overlay Attributes & Global Module References ---------------
    //------------------------------------------------------------------------

protected://fields: overlay attributes

    // overlay identity
    cModule* thisTerminal; //!< pointer to corresponding node
    NodeHandle thisNode;   //!< NodeHandle to this node

    // references to global modules
    BootstrapOracle* bootstrapOracle;           //!< pointer to BootstrapOracle in this node
    GlobalStatistics* globalStatistics; //!< pointer to GlobalStatistics module in this node
    NotificationBoard* notificationBoard;       //!< pointer to NotificationBoard in this node
    UnderlayConfigurator* underlayConfigurator; //!< pointer to UnderlayConfigurator in this node

    // overlay common parameters
    bool debugOutput;           //!< debug output ?
    bool measureNetwInitPhase;  //!< fetch statistics in init phase ?
    bool onlyCommonAPIMessages; //!< drop non-commonAPIMessages ?
    bool useBaseLookup;		//!< use the base lookup class
    bool iterativeLookup;       //!< do iterative instead of recursive lookups
    int localPort;              //!< used UDP-port
    int hopCountMax;            //!< maximum hop count
    int numDropped;             //!< number of dropped packets
    int bytesDropped;           //!< number of dropped bytes
    bool malicious;             //!< true, if node is malicious
    cOutVector delayVector;     //!< statistical output vector for packet-delays
    cOutVector hopCountVector;  //!< statistical output vector for hop-counts

    //------------------------------------------------------------------------
    //--- Initialization & finishing -----------------------------------------
    //------------------------------------------------------------------------

private://methods: cSimpleModule initialization

    /**
     * initializes base-class-attributes
     *
     * @param stage the init stage
     */
    void initialize(int stage);

    /**
     * collects statistical data
     */
    void finish();

protected://methods: overlay initialization


    //  Default number of stages is 6.
    //      *
    //      * Stage 0-4 is needed to initialize OMNet and BaseOverlay.<br>
    //      * Stage 5-X can be used by the overlay implementation.<br>

    /**
     * Sets init stage.
     *
     * @see InitStages.h
     */
    int numInitStages() const;

    /**
     * Initializes derived-class-attributes.<br>
     *
     * Initializes derived-class-attributes, called by
     * BaseOverlay::initialize(). By default this method is called
     * once. If more stages are needed one can overload
     * numInitStages() and add more stages.
     * 
     * @param stage the init stage
     */
    virtual void initializeOverlay( int stage );

    /**
     * collects statistical data in derived class
     */
    virtual void finishOverlay();

    //------------------------------------------------------------------------
    //--- General Overlay Parameters (getter and setters) --------------------
    //------------------------------------------------------------------------

public://methods: getters and setters

    /**
     * Sets node to a malicious state. This method can be overloaded
     * to manage the state change.
     *
     * @param malicious true, if node behaves maliciously
     */
    virtual void setMalicious( bool malicious );

    /**
     * Returns true, if node is malicious.
     * 
     * @return true, if node is malicious.
     */
    bool isMalicious();

    /**
     * Returns the NodeHandle of this node.
     * 
     * @return the NodeHandle of this node.
     */
    NodeHandle& getThisNode();


    //------------------------------------------------------------------------
    //--- Overlay Common API: Key-based Routing ------------------------------
    //------------------------------------------------------------------------

protected://methods: KBR

    /**
     * Routes message through overlay.
     * 
     * The default implementation uses FindNode to determine next
     * hops and a generic greedy routing algorithm provides with
     * SendToKey.
     *
     * @param key destination key
     * @param msg message to route
     * @param hint next hop (usually unused)
     */
    virtual void route( const OverlayKey& key, cMessage* msg,
                        const NodeHandle& hint
                        = NodeHandle::UNSPECIFIED_NODE );



    /**
     * Query if the node knowns the k-neighborhood for a key.
     * 
     * Query if the node knowns the complete k-neighborhood for 
     * the given key. Usually these are all nodes, that are
     * responsible for the key. If there is no k-neighborhood in
     * a specific overlay protocol, this method simply returns
     * true, if the node is responsible for the key.
     *
     * @param key destination key
     * @return bool true, if the node knows the k-neighborhood for key.
     */
    virtual bool isResponsible(const OverlayKey& key);


    /**
     * Calls deliver function in application.
     *
     * Encapsulates messages in KBRdeliver messages and sends them
     * to application.
     * 
     * @param msg delivered message
     */
    void callDeliver( BaseOverlayMessage* msg);

    /**
     * Calls forward function in application
     *
     * Encapsulates messages in KBRforward messages and sends them
     * to application. <br> the message to be sent through the API
     * must be encapsulated in <code>msg</code>.
     * 
     * @param key destination key
     * @param msg message to forward
     * @param hint next hop (usually unused)
     */
    void callForward( OverlayKey& key, BaseRouteMessage* msg,
                      NodeHandle* hint = NULL );


    //------------------------------------------------------------------------
    //--- Message Handlers ---------------------------------------------------
    //------------------------------------------------------------------------

private://methods: message handling

    /**
     * Checks for message type and calls corresponding method.<br>
     *
     * Checks for message type (from UDP/App or selfmessage) and
     * calls corresponding method like route(), get(), put(),
     * remove(), handleTimerEvent(), handleAppMessage() and
     * handleUDPMessage().
     * 
     * @param msg The message to be handled
     */
    void handleMessage(cMessage* msg);


    /**
     * Handles a BaseOverlayMessage<br>
     *
     * Handles BaseOverlayMessages of type OVERLAYSIGNALING, RPC,
     * APPDATA or OVERLAYROUTE.
     * 
     * @param msg The message to be handled
     */
    void handleBaseOverlayMessage(BaseOverlayMessage* msg);

protected://methods: message handling

    /**
     * Processes messages from underlay
     *
     * @param msg Message from UDP
     */
    virtual void handleUDPMessage(BaseOverlayMessage* msg) = 0;

    /**
     * Processes "timer" self-messages
     *
     * @param msg A self-message
     */
    virtual void handleTimerEvent(cMessage* msg);

    /**
     * Processes non-commonAPI messages.<br>
     *
     * Processes non-commonAPI messages if parameter
     * "onlyCommonAPIMessages" is false.
     * 
     * @param msg non-commonAPIMessage
     */
    virtual void handleAppMessage(cMessage* msg);

    /**
     * Collect overlay specific sent messages statistics
     *
     * This method is called from BaseOverlay::sendMessageToUDP()
     * for every overlay message that is sent by a node. Use this
     * to collect statistical data for overlay protocol specific
     * message types.
     *
     * @param msg The overlay message to be sent to the UDP layer
     */
    virtual void recordOverlaySentStats(BaseOverlayMessage* msg);


    //------------------------------------------------------------------------
    //--- Overlay Topology Drawing -------------------------------------------
    //------------------------------------------------------------------------

private://fields: draw overlay topology

    cGate* thisOutGateArray;  //!< out-gate array for vis. overlay topology
    cGate* thisInGateArray;   //!< in-gate array for vis. overlay topology
    bool drawOverlayTopology; //!< draw overlay topology?

    enum Direction { IN, OUT };

    /**
     * compacts arrow gate-array
     *
     * @param terminal node
     * @param dir in- or out-array?
     */
    void compactGateArray(cModule* terminal, enum Direction dir);

protected://methods: icons and ui support

    /**
     * Colors module-icon blue (ready) or red (not ready)
     *
     * @param ready state to visualize
     */
    void setReadyIcon( bool ready );

    /**
     * Draws an arrow from this node to neighbor
     *
     * @param neighbor neighbor to point to
     * @param flush delete all previous drawn arrows starting at this node?
     * @param displayString display string to define the arrow drawing style
     * @todo add neighbor's kind to distinguish arrows pointing to the
     * same neighbor
     */
    void showOverlayNeighborArrow( const NodeHandle& neighbor,
                                   bool flush = true,
                                   char* displayString = NULL );

    /**
     * Removes an arrow from this node to neighbor
     *
     * @param neighbor neighbor to remove arrow to
     * @todo add neighbor's kind to distinguish arrows pointing to the
     * same neighbor
     */
    void deleteOverlayNeighborArrow( const NodeHandle& neighbor );

    //------------------------------------------------------------------------
    //--- Messages -----------------------------------------------------------
    //------------------------------------------------------------------------

public://methods: sending packets over udp

    /**
     * Sends message to underlay
     * 
     * @param dest destination node
     * @param msg message to send
     */
    void sendMessageToUDP( const NodeHandle& dest, BaseOverlayMessage* msg );

    //------------------------------------------------------------------------
    //--- RPC Handling -------------------------------------------------------
    //------------------------------------------------------------------------

private://classes: rpc handling

    class RpcState
    {
    public:
        int id;
        int retries;
        RpcListener* listener;
        NodeHandle dest;
        OverlayKey destKey;
        BaseCallMessage* callMsg;
        RpcTimeoutMessage* timeoutMsg;
        simtime_t timeSent;
    };

private://fields: rpc handling

    typedef hash_map<int,RpcState> RpcStates;

    int rpcsPending;
    RpcListener* defaultRpcListener;
    RpcStates rpcStates;

private://methods: rpc handling

    /**
     * Handles incoming rpc messages and delegates them to the
     * corresponsing listeners or handlers.
     * 
     * @param msg The message to handle.
     */
    void internalHandleRpcMessage( BaseRpcMessage* msg );

    /**
     * Handles internal rpc requests.<br>
     * 
     * This method is used to implement basic functionionality in
     * the BaseOverlay.
     * 
     * @param msg The call message
     * @return bool true, if call has been handled.
     */
    bool internalHandleRpc( BaseCallMessage* msg );

    /**
     * Initializes Remote-Procedure state.
     */
    void initRpcs( );

    /**
     * Deinitializes Remote-Procedure state.
     */
    void finishRpcs( );

public://methods: rpc handling

    /**
     * Sends a Remote-Procedure-Call message to the underlay.<br>
     * 
     * If no timeout is provided, a default value of 1.0 for
     * underlay and 5.0 for a overlay rpc is used. After a timeout
     * the message gets retransmitted for at maximum retries times.
     * 
     * The destKey
     * attribute is kept untouched. 
     * 
     * @param dest Destination node handle
     * @param msg RPC Call Message
     * @param rpcListener RPC Listener
     * @param destKey first route the RPC to the node that is
     *        responsible for kestkey
     * @param rpcId RPC id
     * @param timeout RPC timeout
     * @param retries How often we try to resent rpc call, if it gets lost
     * @return The nonce of the RPC
     */
    uint32_t sendRpcMessage( const NodeHandle& dest, BaseCallMessage* msg,
                             RpcListener* rpcListener = NULL,
                             const OverlayKey& destKey
                             = OverlayKey::UNSPECIFIED_KEY,
                             int rpcId = -1, simtime_t timeout = -1,
                             int retries = 0 );

    /**
     * Cancels a Remote-Procedure-Call.
     * 
     * @param nonce The nonce of the RPC
     */
    void cancelRpcMessage( uint32_t nonce );


protected://methods: rpc handling

    /**
     * Send Remote-Procedure response message and deletes call
     * message.<br>
     * 
     * The destKey attribute is kept untouched. If call or reponse
     * is NULL no reponse will be sent.
     * 
     * @param call The corresponding call message to the reponse
     * @param response The call return value
     */
    void sendRpcResponse( BaseCallMessage* call,
                          BaseResponseMessage* response );

    /**
     * Processes Remote-Procedure-Call invokation messages.<br>
     * 
     * This method should be overloaded when the overlay provides
     * RPC functionality. One can use
     */
    virtual void handleRpc( BaseCallMessage* msg );

    //------------------------------------------------------------------------
    //--- Basic Routing ------------------------------------------------------
    //------------------------------------------------------------------------
protected://fields: config
    BaseLookupConfiguration baseLookupConfig;

    class lookupHashFcn
    {
    public:
        size_t operator()( const AbstractLookup* l1 ) const
        {
            return (size_t)l1;
        }
        bool operator()(const AbstractLookup* l1,
                        const AbstractLookup* l2) const
        {
            return (l1 == l2);
        }
    };

    typedef hash_set<AbstractLookup*, lookupHashFcn, lookupHashFcn> LookupSet;

    LookupSet lookups;

private://methods: internal routing

    void initLookups();
    void finishLookups();

    void internalRoute( const OverlayKey& key, cMessage* msg,
                        const NodeHandle& hint
                        = NodeHandle::UNSPECIFIED_NODE );


public://methods: basic message routing

    /**
     * Sends a message to a overlay node, with the generic routing
     * algorithm.
     * 
     * @param key The destination key
     * @param message Message to be sent
     * @param numNeighbors number of neighbors to send message to
     *                     (multicast)
     * @param nextHop If nextHop is given, the message gets sent to this
     *                node before it is routed (nextHop is used as a proxy)
     */
    void sendToKey( const OverlayKey& key, BaseOverlayMessage* message,
                    uint numNeighbors = 0,
                    const NodeHandle& nextHop = NodeHandle::UNSPECIFIED_NODE);

protected://methods: routing class factory

    /**
     * Creates a abstract iterative lookup instance.
     * 
     * @return AbstractLookup* The new lookup instance.
     */
    virtual AbstractLookup* createLookup();

    /**
     * Removes the abstract lookup instance.
     * 
     * @param lookup
     */
    virtual void removeLookup( AbstractLookup* lookup );

    /**
     * This method should implement the distance between two keys.
     * It may be overloaded to implement a new metric. The default
     * implementation uses the standard-metric d = abs(x-y).
     * 
     * @param x Left-hand-side Key
     * @param y Right-hand-side key
     * @return OverlayKey Distance between x and y
     */
    virtual OverlayKey distance( const OverlayKey& x,
                                 const OverlayKey& y ) const;

    /**
     * Implements the find node call.
     * 
     * This method simply returns the closest nodes known in the
     * corresponding routing topology. If the node is in the
     * k-neighborhood to key (isResponsible(key) = true),
     * this method returns all k neighbors, with the closest
     * neighbor to the key first.
     * 
     * @param key The lookup key.
     * @param msg A pointer to the BaseRouteMessage or FindNodeCall
     *                   message of this lookup.
     * @return NodeVector with closest nodes.
     */
    virtual NodeVector* findNode( const OverlayKey& key,
                                  BaseOverlayMessage* msg = NULL);

    /**
     * Implements a ping call.<br>
     * 
     * This method implementts a simple ping call.
     * 
     * @param call The ping message
     * @return The ping response or NULL if no response should be
     *         sent.
     */
    virtual PingResponse* ping( PingCall* call );

private://methods: rpc stub

    void findNodeRpc( FindNodeCall* call );
    void pingRpc( PingCall* call );

};

#endif
