//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file NodeHandle.cc
 * @author Markus Mauch
 * @author Sebastian Mies
 */

#include <omnetpp.h>

#include <IPAddress.h>

#include "NodeHandle.h"
#include "BaseOverlay.h"

class BaseOverlay;

// predefined node handle
const NodeHandle NodeHandle::UNSPECIFIED_NODE;

std::ostream& operator<<(std::ostream& os, const NodeHandle& n)
{
    if (n.isUnspecified()) {
        os << "<unspec>";
    } else {
        os << n.ip << ":" << n.port << " " << n.key
        << " " << n.moduleId;
    }

    return os;
};


//default-constructor
NodeHandle::NodeHandle()
{
    moduleId = -1;
    port = -1;
    key = OverlayKey::UNSPECIFIED_KEY;
}

//copy constructor
NodeHandle::NodeHandle( const NodeHandle& handle )
{
    moduleId = handle.moduleId;
    key = handle.key;
    port = handle.port;
    ip = handle.ip;
}


//complete constructor
NodeHandle::NodeHandle( const OverlayKey& key,
                        const IPvXAddress& ip, int port,
                        int moduleId )
{
    this->moduleId = moduleId;
    this->ip = ip;
    this->port = port;
    this->key = key;
}

//public
bool NodeHandle::isUnspecified() const
{
    return (moduleId == -1);
}

//public
bool NodeHandle::isMalicious() const
{
    BaseOverlay* overlay
	= dynamic_cast<BaseOverlay*>(simulation.module(moduleId));
    if (overlay==NULL)
        opp_error("NodeHandle not associated with a BaseOverlay Module");
    return overlay->isMalicious();
}

//public
NodeHandle& NodeHandle::operator=(const NodeHandle& rhs)
{
    this->key = rhs.key;
    this->ip = rhs.ip;
    this->port = rhs.port;
    this->moduleId = rhs.moduleId;

    return *this;
}

//public
bool NodeHandle::operator==(const NodeHandle& rhs) const
{
    assertUnspecified( rhs );
    return ( this->moduleId == rhs.moduleId &&
             this->key == rhs.key &&
             this->ip == rhs.ip && this->port == rhs.port );
}

//public
bool NodeHandle::operator!=(const NodeHandle& rhs) const
{
    assertUnspecified( rhs );
    return !( this->moduleId == rhs.moduleId &&
              this->key == rhs.key &&
              this->ip == rhs.ip && this->port == rhs.port );
}

//public
bool NodeHandle::operator<(const NodeHandle &rhs) const
{
    assertUnspecified(rhs);
    return this->key < rhs.key;
}

//public
bool NodeHandle::operator>(const NodeHandle &rhs) const
{
    assertUnspecified(rhs);
    return this->key > rhs.key;
}

//public
bool NodeHandle::operator<=(const NodeHandle &rhs) const
{
    assertUnspecified(rhs);
    return this->key <= rhs.key;
}

//public
bool NodeHandle::operator>=(const NodeHandle &rhs) const
{
    assertUnspecified(rhs);
    return this->key >= rhs.key;
}

//public
void NodeHandle::setAddress( const IPvXAddress& ip, int port )
{
    this->ip = ip;
    if (port!=-1)
        this->port = port;
}

//public
void NodeHandle::setPort( int port )
{
    this->port = port;
}

//public
void NodeHandle::setModuleId( int moduleId )
{
    this->moduleId = moduleId;
}

//public
void NodeHandle::setKey( const OverlayKey& key )
{
    this->key = key;
}

//public
const IPvXAddress& NodeHandle::getAddress() const
{
    return ip;
}

//public
int NodeHandle::getPort() const
{
    return port;
}

//public
int NodeHandle::getModuleId() const
{
    return moduleId;
}

//public
const OverlayKey& NodeHandle::getKey() const
{
    return key;
}

//public
size_t NodeHandle::hash() const
{
    size_t iphash;
    if (ip.isIPv6()) {
        uint32_t* addr = ip.get6().words();
        iphash = (size_t)(addr[0]^addr[1]^addr[2]^addr[3]);
    } else {
        iphash = (size_t)ip.get4().getInt();
    }

    return (size_t)(iphash^port^key.hash());
}


//private
inline void NodeHandle::assertUnspecified( const NodeHandle& handle ) const
{
    if ( this->isUnspecified() || handle.isUnspecified() )
        opp_error("NodeHandle: Trying to compare unspecified nodeHandle!");
}


// Automatically supply array (un)packing functions
template<typename T>
void doPacking(cCommBuffer *b, T *t, int n) {
    for (int i=0; i<n; i++)
        doPacking(b,t[i]);
}
template<typename T>
void doUnpacking(cCommBuffer *b, T *t, int n) {
    for (int i=0; i<n; i++)
        doUnpacking(b,t[i]);
}
inline void doPacking(cCommBuffer *, cPolymorphic&) {}
inline void doUnpacking(cCommBuffer *, cPolymorphic&) {}

#define DOPACKING(T,R) \
    inline void doPacking(cCommBuffer *b, T R a) {b->pack(a);}  \
    inline void doPacking(cCommBuffer *b, T *a, int n) {b->pack(a,n);}  \
    inline void doUnpacking(cCommBuffer *b, T& a) {b->unpack(a);}  \
    inline void doUnpacking(cCommBuffer *b, T *a, int n) {b->unpack(a,n);}
#define _
DOPACKING(char,_)
DOPACKING(unsigned char,_)
DOPACKING(bool,_)
DOPACKING(short,_)
DOPACKING(unsigned short,_)
DOPACKING(int,_)
DOPACKING(unsigned int,_)
DOPACKING(long,_)
DOPACKING(unsigned long,_)
DOPACKING(float,_)
DOPACKING(double,_)
DOPACKING(long double,_)
DOPACKING(char *,_)
DOPACKING(const char *,_)
DOPACKING(opp_string,&)
//DOPACKING(std::string,&)
#undef _
#undef DOPACKING


void NodeHandle::netPack(cCommBuffer *b)
{
    //cMessage::netPack(b);
    doPacking(b,this->ip);
    doPacking(b,this->key);
    doPacking(b,this->port);
    doPacking(b,this->moduleId);
}

void NodeHandle::netUnpack(cCommBuffer *b)
{
    //cMessage::netUnpack(b);
    doUnpacking(b,this->ip);
    doUnpacking(b,this->key);
    doUnpacking(b,this->port);
    doUnpacking(b,this->moduleId);
}
