//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file NodeHandle.h
 * @author Markus Mauch
 * @author Sebastian Mies
 */
#ifndef __NODEHANDLE_H_
#define __NODEHANDLE_H_

class NodeHandle;

#include <stdint.h>
#include <omnetpp.h>
#include <IPvXAddress.h>
#include <NodeVector.h>

#include <ext/hash_set>
#include <ext/hash_map>

using namespace __gnu_cxx;
using namespace std;

#include "OverlayKey.h"


/**
 * This class implements a common node handle.<br>
 *
 * It covers the complete node information, like IP-Address,
 * port, NodeID and some additional flags for Simulation
 * behaviour. The information can be sparse, so parts can be
 * omited by setting the property to an unspecified value.
 * 
 * @author Markus Mauch
 * @author Sebastian Mies
 */
class NodeHandle
{
public:
    class hashFcn
    {
    public:
        size_t operator()( const NodeHandle& h1 ) const
        {
            return h1.hash();
        }
    };

public://collection typedefs

    typedef hash_set<NodeHandle, hashFcn> Set;

public://fields

    IPvXAddress ip;
    OverlayKey key;
    int port;
    int moduleId;

public://construction

    /**
     * Constructs a unspecified NodeHandle
     */
    NodeHandle( );

    /**
     * Copy constructor.
     * 
     * @param handle The NodeHandle to copy
     */
    NodeHandle( const NodeHandle& handle );

    /**
     * Complete constructor.
     * 
     * @param key The OverlayKey 
     * @param ip The IPvXAddress
     * @param port The UDP-Port
     * @param moduleId The simulation module id
     */
    NodeHandle( const OverlayKey& key,
                const IPvXAddress& ip,
                int port,
                int moduleId = -1);

public://static fields

    static const NodeHandle UNSPECIFIED_NODE;

public://methods: delegates to OverlayKey and IPvXAddress

    bool operator==(const NodeHandle& rhs) const;
    bool operator!=(const NodeHandle& rhs) const;
    bool operator< (const NodeHandle& rhs) const;
    bool operator> (const NodeHandle& rhs) const;
    bool operator<=(const NodeHandle& rhs) const;
    bool operator>=(const NodeHandle& rhs) const;

public://methods: operators

    NodeHandle& operator=(const NodeHandle& rhs);

public://methods: setters and getters

    void setAddress( const IPvXAddress& ip, int port = -1 );
    void setPort( int port );
    void setModuleId( int moduleId );
    void setKey( const OverlayKey& key );

    const IPvXAddress& getAddress() const;
    int getPort() const;
    int getModuleId() const;
    const OverlayKey& getKey() const;

    bool isUnspecified() const;
    bool isMalicious() const;

public://methods: hashing

    size_t hash() const;

public://methods: c++ streaming

    friend std::ostream& operator<<(std::ostream& os, const NodeHandle& n);

private://methods:

    void assertUnspecified( const NodeHandle& handle ) const;

public:
    
    void netPack(cCommBuffer *b);
    void netUnpack(cCommBuffer *b);
};


inline void doPacking(cCommBuffer *b, NodeHandle& obj) {obj.netPack(b);}
inline void doUnpacking(cCommBuffer *b, NodeHandle& obj) {obj.netUnpack(b);}


#endif
