//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file OverlayKey.h
 * @author Sebastian Mies
 */

#ifndef __OVERLAYKEY_H_
#define __OVERLAYKEY_H_

#include <string>
#include <INETDefs.h>
#include <assert.h>
#include <gmp.h>
#include <stdint.h>


// replacement function for mpn_random() using omnet's rng
inline void omnet_random(mp_limb_t *r1p, mp_size_t r1n);

/**
 * A common overlay key class.
 *
 * Wraps common functions from Gnu MP library.
 *
 * @author Sebastian Mies.
 */
class OverlayKey
{
public:
    //-------------------------------------------------------------------------
    // constants
    //-------------------------------------------------------------------------

    static const OverlayKey UNSPECIFIED_KEY;
    static const OverlayKey ZERO;
    static const OverlayKey ONE;

    //-------------------------------------------------------------------------
    // construction and destruction
    //-------------------------------------------------------------------------

    /**
     * Default constructor
     *
     * Contructs a unspecified overlay key
     */
    OverlayKey();

    /**
     * Constructs a overlay key initialized with a common integer
     * 
     * @param num The integer to initialize this key with
     */
    OverlayKey( uint32_t num );

    /**
     * Constructs a key out of a string number.
     */
    OverlayKey( const char* str, uint radix = 16 );

    /**
     * Copy contructor.
     * 
     * @param rhs The key to copy.
     */
    OverlayKey( const OverlayKey& rhs );

    /**
     * Default destructor.
     * 
     * Does nothing ATM.
     */
    ~OverlayKey();

    //-------------------------------------------------------------------------
    // string representations & node key attributes
    //-------------------------------------------------------------------------



    /**
     * Returns a string representation of this key
     * 
     * @return String representation of this key
     */
    std::string toString( uint base = 16 ) const;

    /**
     * Common stdc++ console output method
     */
    friend std::ostream& operator<<(std::ostream& os, const OverlayKey& c);

    /**
     * Returns true, if the key is unspecified
     * 
     * @return Returns true, if the key is unspecified
     */
    bool isUnspecified() const;

    //-------------------------------------------------------------------------
    // operators
    //-------------------------------------------------------------------------

    /**
     * Common compare operators
     */
    bool operator< ( const OverlayKey& compKey ) const;
    bool operator> ( const OverlayKey& compKey ) const;
    bool operator<=( const OverlayKey& compKey ) const;
    bool operator>=( const OverlayKey& compKey ) const;
    bool operator==( const OverlayKey& compKey ) const;
    bool operator!=( const OverlayKey& compKey ) const;

    /**
     * Unifies all compare operations in one method
     * 
     * @param compKey key to compare with
     * @return int -1 if smaller, 0 if equal, 1 if greater
     */
    int compareTo( const OverlayKey& compKey ) const;

    /**
     * Assign and arithmetic operators
     */
    OverlayKey& operator= ( const OverlayKey& rhs );
    OverlayKey& operator--();
    OverlayKey& operator++();
    OverlayKey& operator+=( const OverlayKey& rhs );
    OverlayKey& operator-=( const OverlayKey& rhs );

    OverlayKey  operator+ ( const OverlayKey& rhs ) const;
    OverlayKey  operator- ( const OverlayKey& rhs ) const;
    OverlayKey  operator--( int );
    OverlayKey  operator++( int );

    /**
     * Bit operators
     */
    OverlayKey  operator>>( uint num ) const;          // shift right
    OverlayKey  operator<<( uint num ) const;          // shift left
    OverlayKey  operator& ( const OverlayKey& rhs ) const; // bitwise and
    OverlayKey  operator| ( const OverlayKey& rhs ) const; // bitwise or
    OverlayKey  operator^ ( const OverlayKey& rhs ) const; // bitwise xor
    OverlayKey  operator~ () const;            // complement
    bool     operator[]( uint n ) const;           // returns the n-th bit


    //-------------------------------------------------------------------------
    // additional math
    //-------------------------------------------------------------------------

    /**
     * Returns a sub integer at position p with n-bits
     * 
     * @param p the position of the sub-integer
     * @param n the number of bits to be returned (max.32)
     * @return The substring.
     */
    uint32_t get
        (uint p, uint n) const;

    /**
     * Returns a hash value for the key
     * 
     * @return size_t The hash value
     */
    size_t hash() const;

    /**
     * Returns the position of the msb in this key, which represents
     * just the logarithm to base 2.
     * 
     * @return The logartithm to base 2 of this key.
     */
    int log2() const;

    /**
     * Fills the suffix starting at pos with random bits to lsb.
     * 
     * @param pos
     * @return OverlayKey
     */
    OverlayKey randomSuffix(uint pos) const;

    /**
     * Fills the prefix starting at pos with random bits to msb.
     * 
     * @param pos
     * @return OverlayKey
     */
    OverlayKey randomPrefix(uint pos) const;

    /**
     * Returns true, if this key is element of the interval (keyA, keyB)
     * on the ring.
     * 
     * @param keyA The left border of the interval
     * @param keyB The right border of the interval
     * @return True, if the key is element of the interval (keyA, keyB)
     */
    bool isBetween(const OverlayKey& keyA, const OverlayKey& keyB) const;

    /**
     * Returns true, if this key is element of the interval (keyA, keyB]
     * on the ring.
     * 
     * @param keyA The left border of the interval
     * @param keyB The right border of the interval
     * @return True, if the key is element of the interval (keyA, keyB]
     */
    bool isBetweenR(const OverlayKey& keyA, const OverlayKey& keyB) const;

    /**
     * Returns true, if this key is element of the interval [keyA, keyB)
     * on the ring.
     * 
     * @param keyA The left border of the interval
     * @param keyB The right border of the interval
     * @return True, if the key is element of the interval [keyA, keyB)
     */
    bool isBetweenL(const OverlayKey& keyA, const OverlayKey& keyB) const;

    /**
     * Returns true, if this key is element of the interval [keyA, keyB]
     * on the ring.
     * 
     * @param keyA The left border of the interval
     * @param keyB The right border of the interval
     * @return True, if the key is element of the interval [keyA, keyB]
     */
    bool isBetweenLR(const OverlayKey& keyA, const OverlayKey& keyB) const;

    //-------------------------------------------------------------------------
    // static methods
    //-------------------------------------------------------------------------

    /**
     * Set the length of an OverlayKey
     * 
     * @param length keylength in bits
     */
    static void setKeyLength(uint length);

    /**
     * Returns the length in number of bits.
     * 
     * @return The length in number of bits.
     */
    static uint getLength();

    /**
     * Returns a random key.
     * 
     * @return A random key.
     */
    static OverlayKey random();

    /**
     * Returns the maximum key, i.e. a key filled with bit 1
     * 
     * @return The maximum key, i.e. a key filled with bit 1
     */
    static OverlayKey max();

    /**
     * Returns a key with the SHA1 cryptographic hash of a
     * null-terminated string.
     * 
     * @param str A null-terminated string.
     * @return SHA1 of str
     */
    static OverlayKey sha1( char* str );

    /**
     * Returns a key 2^exponent.
     * 
     * @param exponent The exponent.
     * @return Key=2^exponent.
     */
    static OverlayKey pow2(uint exponent);

    /**
     * A pseudo regression test method.
     * Outputs report to standard output.
     */
    static void test();



private:
    // private constants

    static const uint MAX_KEYLENGTH = 160;
    static uint keyLength;
    static uint aSize;
    static mp_limb_t GMP_MSB_MASK;

    // private fields
    bool isUnspec;

    mp_limb_t key[MAX_KEYLENGTH / (8*sizeof(mp_limb_t)) +
              (MAX_KEYLENGTH % (8*sizeof(mp_limb_t))!=0 ? 1 : 0)];

    // private "helper" methods
    void trim();
    void clear();

public:
    
    void netPack(cCommBuffer *b);
    void netUnpack(cCommBuffer *b);
};

inline void doPacking(cCommBuffer *b, OverlayKey& obj) {obj.netPack(b);}
inline void doUnpacking(cCommBuffer *b, OverlayKey& obj) {obj.netUnpack(b);}

#endif
