//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file UnderlayConfigurator.h
 * @author Stephan Krause, Bernhard Heep
 */

#ifndef _UNDERLAYCONFIGURATOR_H_
#define _UNDERLAYCONFIGURATOR_H_


#include <sys/time.h>
#include <time.h>

#include <omnetpp.h>

#include <InitStages.h>


/**
 * Base class for configurators of different underlay models
 *
 * @author Stephan Krause, Bernhard Heep
 */
class UnderlayConfigurator : public cSimpleModule
{
public:

    /**
     * still in initialization phase?
     */
    bool isInit()
    {
        return init;
    };

    struct timeval initFinishedTime; //!< timestamp at end of init phase

protected:

    /**
     * OMNeT number of init stages
     */
    int numInitStages() const;

    /**
     * OMNeT init methods
     */
    virtual void initialize(int stage);

    /**
     * Init method for derived underlay configurators
     */
    virtual void initializeUnderlay(int stage) = 0;

    /**
     * Creates an overlay node
     *
     * @param initialize 
     */
    virtual int createRandomNode(bool initialize) = 0;

    /**
     * Removes randomly chosen overlay nodes from a randomly chosen access net.
     */
    virtual void killRandomNode() = 0;

    /**
     * Migrates randomly chosen overlay nodes from on access net to another.
     */
    virtual void migrateRandomNode() = 0;

    /**
     * Cleans up configurator 
     */
    virtual void finish() = 0;

    /**
     * Sets display string
     */
    virtual void setDisplayString() = 0;

    /**
     * Node mobility simulation
     *
     * @param msg timer-message
     */
    void handleMessage(cMessage* msg);

    std::vector<std::string> channelTypes; //!< possible access types

    bool init; //!< still in initialization phase?
    int initialOverlayTerminalNum; //!< number of overlay terminals at beginning of simulation
    int targetOverlayTerminalNum; //!< final number of overlay terminals
    int overlayTerminalCount; //!< current number of overlay terminals
    bool keepFirstNode;
    int firstNodeId;

private:

    double creationProbability; //!< probability of creating a new overlay terminal
    double migrationProbability; //!< probability of migrating an overlay terminal
    double removalProbability; //!< probability of removing an overlay terminal
    double initialMean; //!< mean of update interval during initalization phase
    double initialDeviation; //!< deviation of update interval during initalization phase
    double targetMean; //!< mean of update interval after initalization phase
    //double targetDeviation; //!< deviation of update interval after initalization phase

    cMessage* churnMsg;
    bool churnIntervalChanged;
};

#endif
