//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file Chord.cc
 * @author Markus Mauch, Ingmar Baumgart
 */

#include <IPAddressResolver.h>
#include <IPvXAddress.h>
#include <InterfaceTable.h>
#include <IPv4InterfaceData.h>

#include <InitStages.h>

#include "Chord.h"


Define_Module(Chord);

void Chord::initializeOverlay(int stage)
{
    // because of IPAddressResolver, we need to wait until interfaces
    // are registered, address auto-assignment takes place etc.
    if(stage != MIN_STAGE_OVERLAY)
        return;

    // fetch some parameters
    successorListSize = par("successorListSize");
    joinRetry = par("joinRetry");
    stabilizeRetry = par("stabilizeRetry");
    joinDelay = par("joinDelay");
    stabilizeDelay = par("stabilizeDelay");
    fixfingersDelay = par("fixfingersDelay");
    aggressiveJoinMode = par("aggressiveJoinMode");

    keyLength = OverlayKey::getLength();
    missingPredecessorStabRequests = 0;
    missingSuccessorStabResponses = 0;

    // statistics
    joinCount = 0;
    stabilizeCount = 0;
    fixfingersCount = 0;
    notifyCount = 0;
    newsuccessorhintCount = 0;
    joinBytesSent = 0;
    stabilizeBytesSent = 0;
    notifyBytesSent = 0;
    fixfingersBytesSent = 0;
    newsuccessorhintBytesSent = 0;


    // find friend modules
    findFriendModules();

    // add some watches
    WATCH(predecessorNode);
    WATCH(thisNode);
    WATCH(bootstrapNode);
    WATCH(joinRetry);
    WATCH(missingPredecessorStabRequests);
    WATCH(missingSuccessorStabResponses);

    // self-messages
    join_timer = new cMessage("join_timer");
    stabilize_timer = new cMessage("stabilize_timer");
    fixfingers_timer = new cMessage("fixfingers_timer");

    // initialize chord protocol
    changeState(INIT);
    changeState(BOOTSTRAP);
}


Chord::~Chord()
{
    // destroy self timer messages
    cancelEvent(join_timer);
    cancelEvent(stabilize_timer);
    cancelEvent(fixfingers_timer);

    delete join_timer;
    delete stabilize_timer;
    delete fixfingers_timer;
}


void Chord::receiveChangeNotification(int category, cPolymorphic* details)
{
    Enter_Method_Silent();
    // get new ip address
    thisNode.ip = IPAddressResolver().addressOf(
                      parentModule()->parentModule()).get4();

    changeState(INIT);
    changeState(BOOTSTRAP);
}


void Chord::changeState(int toState)
{
    //
    // Defines tasks to be executed when a state change occurs.
    //

    switch (toState) {
    case INIT:
        state = INIT;

        // remove current node handle from the bootstrap list
        if(!thisNode.key.isUnspecified()) {
            bootstrapOracle->removePeer(thisNode);
        }

        // Calculate node's id by hashing its IP address
        //  thisNode.key = OverlayKey::sha1(const_cast<char*>(
        //                      thisNode.ip.str().c_str()));
        // better use random numbers (our ip address might not be random)
        // keep old id if INIT gets called twice
        if (thisNode.key.isUnspecified())
            thisNode.key = OverlayKey::random();


        // initialize predecessor pointer
        predecessorNode = NodeHandle::UNSPECIFIED_NODE;

        // initialize finger table and successor list
        initializeFriendModules();

        updateTooltip();

        // debug message
        if (debugOutput) {
            EV << "CHORD: Node " << thisNode.ip
            << " entered INIT stage." << endl;
        }
        // FIXME: bubble() sometimes doesn't work
        parentModule()->parentModule()->bubble("Enter INIT state.");
        break;

    case BOOTSTRAP:
        state = BOOTSTRAP;

        // initiate bootstrap process
        cancelEvent(join_timer);
        // workaround: prevent notificationBoard from taking
        // ownership of join_timer message
        take(join_timer);
        scheduleAt(simulation.simTime(), join_timer);

        // debug message
        if (debugOutput) {
            EV << "CHORD: Node " << thisNode.ip
            << " entered BOOTSTRAP stage." << endl;
        }
        parentModule()->parentModule()->bubble("Enter BOOTSTRAP state.");

        // find a new bootstrap node and enroll to the bootstrap list
        bootstrapNode = bootstrapOracle->getBootstrapNode();

        // is this the first node?
        if (bootstrapNode.isUnspecified()) {
            // create new cord ring
            bootstrapNode = thisNode;
            changeState(READY);
            updateTooltip();
        }
        break;

    case READY:
        state = READY;

        bootstrapOracle->registerPeer(thisNode);

        // initiate stabilization protocol
        cancelEvent(stabilize_timer);
        scheduleAt(simulation.simTime() + stabilizeDelay, stabilize_timer);

        // initiate finger repair protocol
        cancelEvent(fixfingers_timer);
        scheduleAt(simulation.simTime() + fixfingersDelay,
                   fixfingers_timer);

        // debug message
        if (debugOutput) {
            EV << "CHORD: Node " << thisNode.ip << " entered READY stage."
            << endl;
        }
        parentModule()->parentModule()->bubble("Enter READY state.");
        break;
    }

    setReadyIcon(state == READY);
}


void Chord::handleTimerEvent(cMessage* msg)
{
    // catch JOIN timer
    if (msg->isName("join_timer")) {
        handleJoinTimerExpired(msg);
    }
    // catch STABILIZE timer
    else if (msg->isName("stabilize_timer")) {
        handleStabilizeTimerExpired(msg);
    }
    // catch FIX_FINGERS timer
    else if (msg->isName("fixfingers_timer")) {
        handleFixFingersTimerExpired(msg);
    }
    // unknown self message
    else {
        error("Chord::handleTimerEvent(): received self message of "
              "unknown type!");
    }
}


void Chord::handleUDPMessage(BaseOverlayMessage* msg)
{
    ChordMessage* chordMsg = check_and_cast<ChordMessage*>(msg);
    switch(chordMsg->getCommand()) {
    case NEWSUCCESSORHINT:
        handleNewSuccessorHint(chordMsg);
        break;
    default:
        error("handleUDPMessage(): Unknown message type!");
        break;
    }

    delete chordMsg;
}


void Chord::handleRpc(BaseCallMessage* msg)
{
    if (state != READY) {
        delete msg;
        EV << "Chord::handleRpc(): Received RPC call "
        << "and state != READY!" << endl;
        return;
    }

    // delegate messages
    RPC_SWITCH_START( msg )
    // RPC_DELEGATE( <messageName>[Call|Response], <methodToCall> )
    RPC_DELEGATE( Join, rpcJoin );
    RPC_DELEGATE( Notify, rpcNotify );
    RPC_DELEGATE( Stabilize, rpcStabilize );
    RPC_DELEGATE( Fixfingers, rpcFixfingers );
    RPC_SWITCH_END( )
}

void Chord::handleRpcResponse(BaseResponseMessage* msg, int rpcId,
                              simtime_t rtt)
{
    RPC_SWITCH_START(msg)
    RPC_ON_RESPONSE( Join ) {
        handleRpcJoinResponse(_JoinResponse);
        EV << "Join RPC Response received: id=" << rpcId
        << " msg=" << *_JoinResponse << " rtt=" << rtt << endl;
        break;
    }
    RPC_ON_RESPONSE( Notify ) {
        handleRpcNotifyResponse(_NotifyResponse);
        EV << "Notify RPC Response received: id=" << rpcId
        << " msg=" << *_NotifyResponse << " rtt=" << rtt << endl;
        break;
    }
    RPC_ON_RESPONSE( Stabilize ) {
        handleRpcStabilizeResponse(_StabilizeResponse);
        EV << "Stabilize RPC Response received: id=" << rpcId
        << " msg=" << *_StabilizeResponse << " rtt=" << rtt << endl;
        break;
    }
    RPC_ON_RESPONSE( Fixfingers ) {
        handleRpcFixfingersResponse(_FixfingersResponse);
        EV << "Fixfingers RPC Response received: id=" << rpcId
        << " msg=" << *_FixfingersResponse << " rtt=" << rtt << endl;
        break;
    }
    RPC_SWITCH_END( )
}

void Chord::handleRpcTimeout(BaseCallMessage* msg, const NodeHandle& dest,
                             int rpcId)
{
    RPC_SWITCH_START(msg)
    RPC_ON_CALL( FindNode ) {
        EV << "FindNode RPC Call timed out: id=" << rpcId
        << " msg=" << *_FindNodeCall << endl;
        break;
    }
    RPC_ON_CALL( Join ) {
        EV << "Join RPC Call timed out: id=" << rpcId
        << " msg=" << *_JoinCall << endl;
        break;
    }
    RPC_ON_CALL( Notify ) {
        EV << "Notify RPC Call timed out: id=" << rpcId
        << " msg=" << *_NotifyCall << endl;
        break;
    }
    RPC_ON_CALL( Stabilize ) {
        EV << "Stabilize RPC Call timed out: id=" << rpcId
        << " msg=" << *_StabilizeCall << endl;
        break;
    }
    RPC_ON_CALL( Fixfingers ) {
        EV << "Fixfingers RPC Call timed out: id=" << rpcId
        << " msg=" << *_FixfingersCall << endl;
        break;
    }
    RPC_SWITCH_END( )
}


bool Chord::isResponsible(const OverlayKey& key)
{
    if (key.isUnspecified())
        error("Chord::isResponsible(): key is unspecified!");

    if (state != READY)
        return false;

    // if this is the first and only node on the ring, it is responsible
    if (predecessorNode.isUnspecified()) {
        if(successorList->isEmpty()) {
            return true;
        } else {
            return false;
        }
    }

    // is the message destined for this node?
    if (key.isBetweenR(predecessorNode.key, thisNode.key)) {
        return true;
    }

    return false;
}


NodeVector* Chord::findNode(const OverlayKey& key, BaseOverlayMessage* msg)
{
    NodeVector* nextHop = new NodeVector(1);

    if (state != READY)
        return nextHop;

//     // example code for findNodeExt

//      if (msg != NULL) {
//      	if (!msg->hasObject("findNodeExt")) {
//      	    ChordFindNodeExtMessage *extMsg =
//      		new ChordFindNodeExtMessage("findNodeExt");
//        		extMsg->setLength(8*10);
//      	    msg->addObject( extMsg );
//     	         }
//
//       	ChordFindNodeExtMessage *extMsg =
//       	    (ChordFindNodeExtMessage*) msg->getObject("findNodeExt");
//
// 	    cout << "ChordCount: " << extMsg->getChordCount() + 1 << endl;
//
//     	    extMsg->setChordCount(extMsg->getChordCount() + 1);
//      }

    // if key is unspecified, the message is for this node
    if (key.isUnspecified()) {
//        nextHop->push_back(thisNode);
        nextHop->push_back(thisNode);
    } else if (isResponsible(key)) {
        // the message is destined for this node
//        nextHop->push_back(thisNode);
        nextHop->push_back(thisNode);
    } else if (key.isBetweenR(thisNode.key,
                              successorList->getSuccessor().key)) {
        // the message destined for our successor
//        nextHop->push_back(successorList->getSuccessor());
        nextHop->push_back(successorList->getSuccessor());
    } else {
        // find next hop with finger table
        NodeHandle tmpNode = closestPreceedingNode(key);
        if (!tmpNode.isUnspecified()) {
//            nextHop->push_back(tmpNode);
            nextHop->push_back(tmpNode);
        }
    }

    return nextHop;
}


const NodeHandle& Chord::closestPreceedingNode(const OverlayKey& key)
{
    for (int i = fingerTable->getSize() - 1; i >= 0; i--) {
        if (fingerTable->getFinger(i).key.isBetween(thisNode.key, key)) {
            // is there a closer preceeding node in the successor list?
            for (int j = successorList->getSize() - 1; j >= 0; j--) {
                if (successorList->getSuccessor(j).key.
                        isBetween(fingerTable->getFinger(i).key, key)) {
                    return successorList->getSuccessor(j);
                }
            }

            // if no, settle with the node already found
            return fingerTable->getFinger(i);
        }
    }

    // if no finger is found lookup the rest of the successor list
    for (int i = successorList->getSize()-1; i >= 0; i--) {
        if(successorList->getSuccessor(i).key.isBetween(thisNode.key, key)) {
            return successorList->getSuccessor(i);
        }
    }

    // if this is the first and only node on the ring, it is responsible
    if ((predecessorNode.isUnspecified()) &&
            (successorList->getSuccessor() == thisNode)) {
        return thisNode;
    }

    // if there is still no node found return NodeHandle::UNSPECIFIED_NODE
    return NodeHandle::UNSPECIFIED_NODE;
}

void Chord::recordOverlaySentStats(BaseOverlayMessage* msg)
{
    BaseOverlayMessage* innerMsg;

    if (msg->getType() == OVERLAYROUTE)
        innerMsg = dynamic_cast<BaseOverlayMessage*>(msg->encapsulatedMsg());
    else
        innerMsg = msg;

    switch (innerMsg->getType()) {

    case OVERLAYSIGNALING: {
            ChordMessage* chordMsg = dynamic_cast<ChordMessage*>(innerMsg);
            switch(chordMsg->getCommand()) {
            case NEWSUCCESSORHINT:
                RECORD_STATS(newsuccessorhintCount++; newsuccessorhintBytesSent +=
                                 msg->byteLength());
                break;
            }
            break;
        }

    case RPC: {
            if ((dynamic_cast<StabilizeCall*>(innerMsg) != NULL) ||
                    (dynamic_cast<StabilizeResponse*>(innerMsg) != NULL)) {
                RECORD_STATS(stabilizeCount++; stabilizeBytesSent +=
                                 msg->byteLength());
            } else if ((dynamic_cast<NotifyCall*>(innerMsg) != NULL) ||
                       (dynamic_cast<NotifyResponse*>(innerMsg) != NULL)) {
                RECORD_STATS(notifyCount++; notifyBytesSent +=
                                 msg->byteLength());
            } else if ((dynamic_cast<FixfingersCall*>(innerMsg) != NULL) ||
                       (dynamic_cast<FixfingersResponse*>(innerMsg) != NULL)) {
                RECORD_STATS(fixfingersCount++; fixfingersBytesSent +=
                                 msg->byteLength());
            } else if ((dynamic_cast<JoinCall*>(innerMsg) != NULL) ||
                       (dynamic_cast<JoinResponse*>(innerMsg) != NULL)) {
                RECORD_STATS(joinCount++; joinBytesSent += msg->byteLength());
            }
            break;
        }
    }
}


void Chord::finishOverlay()
{
    recordScalar("Chord: Sent JOIN Messages", joinCount);
    recordScalar("Chord: Sent NEWSUCCESSORHINT Messages",
                 newsuccessorhintCount);
    recordScalar("Chord: Sent STABILIZE Messages", stabilizeCount);
    recordScalar("Chord: Sent NOTIFY Messages", notifyCount);
    recordScalar("Chord: Sent FIX_FINGERS Messages", fixfingersCount);
    recordScalar("Chord: Sent JOIN Bytes", joinBytesSent);
    recordScalar("Chord: Sent NEWSUCCESSORHINT Bytes",
                 newsuccessorhintBytesSent);
    recordScalar("Chord: Sent STABILIZE Bytes", stabilizeBytesSent);
    recordScalar("Chord: Sent NOTIFY Bytes", notifyBytesSent);
    recordScalar("Chord: Sent FIX_FINGERS Bytes", fixfingersBytesSent);

    // remove this node from the bootstrap list
    bootstrapOracle->removePeer(thisNode);
}



void Chord::handleJoinTimerExpired(cMessage* msg)
{
    // only process timer, if node is not bootstrapped yet
    if (state == READY)
        return;

    // enter state BOOTSTRAP
    if (state != BOOTSTRAP)
        changeState(BOOTSTRAP);

    // change bootstrap node from time to time
    joinRetry--;
    if (joinRetry == 0) {
        joinRetry = par("joinRetry");
        changeState(BOOTSTRAP);
        return;
    }

    // call JOIN RPC
    JoinCall* call = new JoinCall("JoinCall");
    call->setLength(JOINCALL_L(call));

    sendRpcMessage(bootstrapNode, call, NULL, thisNode.key, -1, joinDelay);

    // schedule next bootstrap process in the case this one fails
    cancelEvent(join_timer);
    scheduleAt(simulation.simTime() + joinDelay, msg);
}


void Chord::handleStabilizeTimerExpired(cMessage* msg)
{
    if (state != READY)
        return;

    if (missingPredecessorStabRequests >= stabilizeRetry) {
        // predecessor node seems to be dead
        // remove it from the predecessor / successor lists
        successorList->removeSuccessor(predecessorNode);
        predecessorNode = NodeHandle::UNSPECIFIED_NODE;

        missingPredecessorStabRequests = 0;
        updateTooltip();
    }

    if (missingSuccessorStabResponses >= stabilizeRetry) {
        // successor node seems to be dead
        // remove it from the predecessor / successor list
        NodeHandle successor = successorList->popSuccessor();

        // if we had a ring consisting of 2 nodes and our successor seems
        // to be dead. Remove also predecessor because the successor
        // and predecessor are the same node
        if ((!predecessorNode.isUnspecified()) &&
                predecessorNode == successor) {
            predecessorNode = NodeHandle::UNSPECIFIED_NODE;
        }

        missingSuccessorStabResponses = 0;
        updateTooltip();

        if (successorList->isEmpty()) {
            changeState(INIT);
            changeState(BOOTSTRAP);
            return;
        }
    }

    if (!successorList->isEmpty()) {
        // call STABILIZE RPC
        StabilizeCall* call = new StabilizeCall("StabilizeCall");
        call->setLength(STABILIZECALL_L(call));

        sendRpcMessage(successorList->getSuccessor(), call);

        missingPredecessorStabRequests++;
        missingSuccessorStabResponses++;
    }

    // schedule next stabilization process
    cancelEvent(stabilize_timer);
    scheduleAt(simulation.simTime() + stabilizeDelay, msg);
}


void Chord::handleFixFingersTimerExpired(cMessage* msg)
{
    if ((state != READY) || successorList->isEmpty())
        return;

    for (uint nextFinger = 0; nextFinger < thisNode.key.getLength();
            nextFinger++) {
        // calculate "n + 2^(i - 1)"
        OverlayKey offset = OverlayKey::pow2(nextFinger);
        OverlayKey lookupKey = thisNode.key + offset;

        // send message only for non-trivial fingers
        if (offset > successorList->getSuccessor().key - thisNode.key) {
            // call FIXFINGER RPC
            FixfingersCall* call = new FixfingersCall("FixfingersCall");
            call->setFinger(nextFinger);
            call->setLength(FIXFINGERSCALL_L(call));

            sendRpcMessage(NodeHandle::UNSPECIFIED_NODE, call, NULL,
                           lookupKey, -1, fixfingersDelay);

        } else {
            // let trivial fingers point to the successor node
            fingerTable->setFinger(nextFinger,
                                   successorList->getSuccessor());
        }
    }

    // schedule next finger repair process
    cancelEvent(fixfingers_timer);
    scheduleAt(simulation.simTime() + fixfingersDelay, msg);
}


void Chord::handleNewSuccessorHint(ChordMessage* chordMsg)
{
    NewSuccessorHintMessage* newSuccessorHintMsg =
        check_and_cast<NewSuccessorHintMessage*>(chordMsg);

    // fetch the successor's predecessor
    NodeHandle predecessor = newSuccessorHintMsg->getPreNode();

    // is the successor's predecessor a new successor for this node?
    if (predecessor.key.isBetween(thisNode.key,
                                  successorList->getSuccessor().key)
            || (thisNode.key == successorList->getSuccessor().key)) {
        // add the successor's predecessor to the successor list
        successorList->addSuccessor(predecessor);
        updateTooltip();
    }
}


void Chord::rpcJoin(JoinCall* joinCall)
{
    NodeHandle requestor = joinCall->getSrcNode();

    // compile successor list
    JoinResponse* joinResponse =
        new JoinResponse("JoinResponse");

    int sucNum = successorList->getSize();
    joinResponse->setSucNum(sucNum);
    joinResponse->setSucNodeArraySize(sucNum);

    for (int k = 0; k < sucNum; k++) {
        joinResponse->setSucNode(k, successorList->getSuccessor(k));
    }

    // sent our predecessor as hint to the joining node
    if (predecessorNode.isUnspecified() && successorList->isEmpty()) {
        // we are the only node in the ring
        joinResponse->setPreNode(thisNode);
    } else {
        joinResponse->setPreNode(predecessorNode);
    }

    joinResponse->setLength(JOINRESPONSE_L(joinResponse));

    sendRpcResponse(joinCall, joinResponse);

    if (aggressiveJoinMode) {
        // aggressiveJoinMode differs from standard join operations:
        // 1. set our predecessor pointer to the joining node
        // 2. send our old predecessor as hint in JoinResponse msgs
        // 3. send a NEWSUCCESSORHINT to our old predecessor to update
        //    its successor pointer

        // send NEWSUCCESSORHINT to our old predecessor

        if (!predecessorNode.isUnspecified()) {
            NewSuccessorHintMessage* newSuccessorHintMsg =
                new NewSuccessorHintMessage("NEWSUCCESSORHINT");
            newSuccessorHintMsg->setCommand(NEWSUCCESSORHINT);

            newSuccessorHintMsg->setSrcNode(thisNode);
            newSuccessorHintMsg->setPreNode(requestor);
            newSuccessorHintMsg->setLength(
                NEWSUCCESSORHINT_L(newSuccessorHintMsg));

            sendMessageToUDP(predecessorNode, newSuccessorHintMsg);
        }

        // the requestor is our new predecessor
        predecessorNode = requestor;
    }

    // if we don't have a successor, the requestor is also our new successor
    if (successorList->isEmpty())
        successorList->addSuccessor(requestor);

    updateTooltip();
}

void Chord::handleRpcJoinResponse(JoinResponse* joinResponse)
{
    // determine the numer of successor nodes to add
    int sucNum = successorListSize - 1;

    if (joinResponse->getSucNum() < successorListSize - 1) {
        sucNum = joinResponse->getSucNum();
    }

    successorList->addSuccessor(joinResponse->getSrcNode());

    // add successor node(s)
    for (int k = 0; k < sucNum; k++) {
        NodeHandle successor = joinResponse->getSucNode(k);
        successorList->addSuccessor(successor);
    }

    // the sender of this message is our new successor
    successorList->addSuccessor(joinResponse->getSrcNode());

    // in aggressiveJoinMode: use hint in JoinResponse
    // to set our new predecessor
    if (aggressiveJoinMode) {
        predecessorNode = joinResponse->getPreNode();
    }

    updateTooltip();

    changeState(READY);

    // immediate stabilization protocol
    cancelEvent(stabilize_timer);
    scheduleAt(simulation.simTime(), stabilize_timer);

    // immediate finger repair protocol
    cancelEvent(fixfingers_timer);
    scheduleAt(simulation.simTime(), fixfingers_timer);
}


void Chord::rpcStabilize(StabilizeCall* call)
{
    // our predecessor seems to be alive
    missingPredecessorStabRequests = 0;

    // reply with StabilizeResponse message
    StabilizeResponse* stabilizeResponse =
        new StabilizeResponse("StabilizeResponse");
    stabilizeResponse->setPreNode(predecessorNode);
    stabilizeResponse->setLength(STABILIZERESPONSE_L(stabilizeResponse));

    sendRpcResponse(call, stabilizeResponse);
}

void Chord::handleRpcStabilizeResponse(StabilizeResponse* stabilizeResponse)
{
    // our successor seems to be alive
    missingSuccessorStabResponses = 0;

    // fetch the successor's predecessor
    NodeHandle predecessor = stabilizeResponse->getPreNode();

    // is the successor's predecessor a new successor for this node?
    if (successorList->isEmpty() ||
            predecessor.key.isBetween(thisNode.key,
                                      successorList->getSuccessor().key)) {
        // add the successor's predecessor to the successor list
        successorList->addSuccessor(predecessor);
        updateTooltip();
    }

    // compile NOTIFY RPC
    NotifyCall* notifyCall = new NotifyCall("NotifyCall");
    notifyCall->setLength(NOTIFYCALL_L(notifyCall));

    sendRpcMessage(successorList->getSuccessor(), notifyCall);
}


void Chord::rpcNotify(NotifyCall* call)
{
    // our predecessor seems to be alive
    missingPredecessorStabRequests = 0;

    NodeHandle predecessor = call->getSrcNode();

    // is the new predecessor closer than the current one?
    if (predecessorNode.isUnspecified() ||
            predecessor.key.isBetween(predecessorNode.key, thisNode.key)) {
        // set up new predecessor
        predecessorNode = predecessor;
        updateTooltip();
    }

    // compile NOTIFY response
    NotifyResponse* notifyResponse = new NotifyResponse("NotifyResponse");

    int sucNum = successorList->getSize();
    notifyResponse->setSucNum(sucNum);
    notifyResponse->setSucNodeArraySize(sucNum);

    for (int k = 0; k < sucNum; k++) {
        notifyResponse->setSucNode(k, successorList->getSuccessor(k));
    }

    notifyResponse->setLength(NOTIFYRESPONSE_L(notifyResponse));

    sendRpcResponse(call, notifyResponse);
}


void Chord::handleRpcNotifyResponse(NotifyResponse* notifyResponse)
{
    if (successorList->getSuccessor() != notifyResponse->getSrcNode()) {
        EV << "Chord::handleRpcNotifyResponse: The srcNode of the received "
        << "NotifyResponse is not our current successor!" << endl;
        return;
    }

    // determine number of successor nodes to add
    int sucNum = successorListSize - 1;
    if (notifyResponse->getSucNum() < successorListSize - 1) {
        sucNum = notifyResponse->getSucNum();
    }

    // replace our successor list by our successor's successor list
    // and add our current successor to the list
    successorList->clear();
    successorList->addSuccessor(notifyResponse->getSrcNode());
    for (int k = 0; k < sucNum; k++) {
        NodeHandle successor = notifyResponse->getSucNode(k);
        // don't add nodes, if this would change our successor
        if (!successor.key.isBetweenLR(thisNode.key,
                                       notifyResponse->getSrcNode().key)) {
            successorList->addSuccessor(successor);
        }
    }
    updateTooltip();
}


void Chord::rpcFixfingers(FixfingersCall* call)
{
    FixfingersResponse* fixfingersResponse =
        new FixfingersResponse("FixfingersResponse");

    fixfingersResponse->setSucNode(thisNode);
    fixfingersResponse->setFinger(call->getFinger());
    fixfingersResponse->setLength(FIXFINGERSRESPONSE_L(fixfingersResponse));

    sendRpcResponse(call, fixfingersResponse);
}


void Chord::handleRpcFixfingersResponse(FixfingersResponse* fixfingersResponse)
{
    // set new finger pointer
    NodeHandle successor = fixfingersResponse->getSucNode();
    fingerTable->setFinger(fixfingersResponse->getFinger(), successor);
}


void
Chord::findFriendModules()
{
    fingerTable = check_and_cast<FingerTable*>
                  (parentModule()->submodule("fingerTable"));

    successorList = check_and_cast<SuccessorList*>
                    (parentModule()->submodule("successorList"));
}


void
Chord::initializeFriendModules()
{
    // initialize finger table
    fingerTable->initializeTable(thisNode.key.getLength(), thisNode);

    // initialize successor list
    successorList->initializeList(par("successorListSize"), thisNode);
}


void Chord::updateTooltip()
{
    if (ev.isGUI()) {
        std::stringstream ttString;

        // show our predecessor and successor in tooltip
        ttString << predecessorNode << endl << thisNode << endl
        << successorList->getSuccessor();

        parentModule()->parentModule()->displayString().
        setTagArg("tt", 0, ttString.str().c_str());
        parentModule()->displayString().
        setTagArg("tt", 0, ttString.str().c_str());
        displayString().setTagArg("tt", 0, ttString.str().c_str());

        // draw an arrow to our current successor
        showOverlayNeighborArrow(successorList->getSuccessor(), true,
                                 "m=m,50,0,50,0;o=red,1");
        showOverlayNeighborArrow(predecessorNode, false,
                                 "m=m,50,100,50,100;o=green,1");
    }
}
