//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file Chord.h
 * @author Markus Mauch, Ingmar Baumgart
 */

#ifndef __CHORD_H_
#define __CHORD_H_

#include <vector>

#include <omnetpp.h>
#include <IPvXAddress.h>

#include <OverlayKey.h>
#include <NodeHandle.h>
#include <BaseOverlay.h>
#include <RpcMacros.h>

#include "ChordMessage_m.h"
#include "FingerTable.h"
#include "SuccessorList.h"


/**
 * protocol states
 */
enum
{
    INIT,
    BOOTSTRAP,
    READY,
};

/**
 * Chord overlay module
 *
 * Implementation of the Chord KBR overlay as described in
 * "Chord: A Scalable Peer-to-Peer Lookup Protocol for Inetnet
 * Applications" by I. Stoica et al. published in Transactions on Networking.
 *
 * @author Markus Mauch, Ingmar Baumgart
 * @see BaseOverlay, FingerTable, SuccessorList
 */
class Chord : public BaseOverlay
{
  public:

    virtual ~Chord();

    virtual void initializeOverlay(int stage);

    /**
     * callback-method for events at the NotificationBoard
     *
     * @param category
     * @param details
     * @todo parameter description, implementation
     */
    virtual void receiveChangeNotification(int category,
                                           cPolymorphic* details);

    /**
     * Query if the node is responsible for a key.
     * 
     * Query if the node currently is responsible for the given
     * key. Usually this means, that the nodeId of this node is close
     * to the key.
     *
     * @param key destination key
     * @return bool true, if the node is responsible for the key.
     */
    virtual bool isResponsible(const OverlayKey& key);

    /**
     * handles self-messages
     *
     * @param msg the self-message
     */
    virtual void handleTimerEvent(cMessage* msg);

    /**
     * processes messages from underlay
     *
     * @param msg message from UDP
     */
    virtual void handleUDPMessage(BaseOverlayMessage* msg);

    virtual void recordOverlaySentStats(BaseOverlayMessage* msg);

    /**
     * collects statisticts
     */
    virtual void finishOverlay();

    /**
     * updates information shown in tk-environment
     */
    virtual void updateTooltip();

  protected:
    int joinRetry; /**< */
    int stabilizeRetry; /**< // retries before neighbor considered failed */
    double joinDelay; /**< */
    double stabilizeDelay; /**< stabilize interval (secs) */
    double fixfingersDelay; /**< */
    int successorListSize; /**< */
    bool aggressiveJoinMode; /**< use modified (faster) JOIN protocol */

    // timer messages
    cMessage* join_timer; /**< */
    cMessage* stabilize_timer; /**< */
    cMessage* fixfingers_timer; /**< */

    // statistics
    int joinCount; /**< */
    int stabilizeCount; /**< */
    int fixfingersCount; /**< */
    int notifyCount; /**< */
    int newsuccessorhintCount; /**< */
    int joinBytesSent; /**< */
    int stabilizeBytesSent; /**< */
    int notifyBytesSent; /**< */
    int fixfingersBytesSent; /**< */
    int newsuccessorhintBytesSent; /**< */


    int state; /**< current node state*/
    int keyLength; /**< length of an overlay key in bits */
    int missingPredecessorStabRequests; /**< missing StabilizeCall msgs */
    int missingSuccessorStabResponses; /**< missing StabilizeResponse msgs */

    /**
     * changes node state
     *
     * @param toState state to change to
     */
    virtual void changeState(int toState);

    // node references
    NodeHandle predecessorNode; /**< predecessor of this node */
    NodeHandle bootstrapNode; /**< node used to bootrap */

    // module references
    FingerTable* fingerTable; /**< pointer to this node's finger table */
    SuccessorList* successorList; /**< pointer to this node's successor list */

    // chord routines

    /**
     * handle a expired join timer
     *
     * @param msg the timer self-message
     */
    virtual void handleJoinTimerExpired(cMessage* msg);

    /**
     * handle a expired stabilize timer
     *
     * @param msg the timer self-message
     */
    virtual void handleStabilizeTimerExpired(cMessage* msg);

    /**
     * handle a expired fix_fingers timer
     *
     * @param msg the timer self-message
     */
    virtual void handleFixFingersTimerExpired(cMessage* msg);

    /**
     * handle a received NEWSUCCESSORHINT message
     *
     * @param chordMsg the message to process
     */
    virtual void handleNewSuccessorHint(ChordMessage* chordMsg);


    /**
     * looks up the finger table and returns the closest preceeding node.
     *
     * @param key key to find the closest preceeding node for
     * @return node handle of the closest preceeding node to key
     */
    virtual const NodeHandle& closestPreceedingNode(const OverlayKey& key);


    /**
     * Assigns the finger table and succesesor list module to our reference
     */
    virtual void findFriendModules();

    /**
     * initializes finger table and successor list
     */
    virtual void initializeFriendModules();

    /**
     * Processes Remote-Procedure-Call invokation messages
     */
    void handleRpc( BaseCallMessage* msg );

    /**
     * Implements the find node call.
     * 
     * This method simply returns the closest nodes known in the
     * corresponding routing topology.
     * 
     * @param key The lookup key.
     * @param msg A pointer to the BaseRouteMessage or FindNodeCall
     *                   message of this lookup.
     * @return NodeVector with closest nodes.
     */
    NodeVector* findNode( const OverlayKey& key, BaseOverlayMessage* msg);

    /**
     * Fixfingers Remote-Procedure-Call
     * 
     * @param call RPC Parameter Message
     */
    void rpcFixfingers( FixfingersCall* call );

    /**
     * Join Remote-Procedure-Call
     * 
     * @param call RPC Parameter Message
     */
    void rpcJoin( JoinCall* call );

    /**
     * NOTIFY Remote-Procedure-Call
     * 
     * @param call RPC Parameter Message
     */
    void rpcNotify( NotifyCall* call );

    /**
     * STABILIZE Remote-Procedure-Call
     * 
     * @param call RPC Parameter Message
     */
    void rpcStabilize( StabilizeCall* call );

    /**
     * This method is called if an RPC response has been received.
     * 
     * @param msg The response message.
     * @param rpcId The RPC id.
     * @param rtt The Round-Trip-Time of this RPC
     */
    void handleRpcResponse( BaseResponseMessage* msg, int rpcId,
                            simtime_t rtt );

    /**
     * This method is called if an RPC timeout has been reached.
     * 
     * @param msg The original RPC message.
     * @param dest The destination of the RPC 
     * @param rpcId The RPC id.
     */
    void handleRpcTimeout(BaseCallMessage* msg, const NodeHandle& dest,
                          int rpcId);

    void handleRpcJoinResponse(JoinResponse* joinResponse);
    void handleRpcNotifyResponse(NotifyResponse* notifyResponse);
    void handleRpcStabilizeResponse(StabilizeResponse* stabilizeResponse);
    void handleRpcFixfingersResponse(FixfingersResponse* fixfingersResponse);
};

#endif
