//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file SuccessorList.cc
 * @author Markus Mauch, Ingmar Baumgart
 */

#include <omnetpp.h>
#include <algorithm>

#include <NodeHandle.h>

#include "SuccessorList.h"


Define_Module(SuccessorList);

//todo: how to delete successorList and what does WATCH_MAP do?
void SuccessorList::initialize(int stage)
{
    // because of IPAddressResolver, we need to wait until interfaces
    // are registered, address auto-assignment takes place etc.
    if (stage != MIN_STAGE_OVERLAY)
        return;

    WATCH_MAP(successorMap);
}

void SuccessorList::handleMessage(cMessage* msg)
{
    error("this module doesn't handle messages, it runs only in initialize()");
}

void SuccessorList::initializeList(uint size, NodeHandle owner)
{
    successorMap.clear();
    successorListSize = size;
    thisNode = owner;
    addSuccessor(thisNode);
}

void SuccessorList::clear()
{
    successorMap.clear();
}


uint SuccessorList::getSize()
{
    return successorMap.size();
}

bool SuccessorList::isEmpty()
{
    if (successorMap.size() == 1 && getSuccessor() == thisNode)
        return true;
    else
        return false;
}

const NodeHandle& SuccessorList::getSuccessor(uint pos)
{
    // check boundaries
    if (pos == 0 && successorMap.size() == 0)
        return NodeHandle::UNSPECIFIED_NODE;

    if (pos >= successorMap.size()) {
        error("Index out of bound (SuccessorList, getSuccessor())");
    }

    std::map<OverlayKey, NodeHandle>::iterator it = successorMap.begin();
    for (uint i= 0; i < pos; i++) {
        it++;
        if (i == (pos-1))
            return it->second;
    }
    return it->second;
}

void SuccessorList::addSuccessor(NodeHandle successor)
{
    OverlayKey sum = successor.key - (thisNode.key + OverlayKey::ONE);
    // perhaps this step can be optimized
    successorMap.insert(make_pair(sum, successor));
    if (successorMap.size() > successorListSize) {
        std::map<OverlayKey, NodeHandle>::iterator it = successorMap.end();
        it--;
        successorMap.erase(it);
    }
}

// todo: how to deal with references
NodeHandle SuccessorList::popSuccessor()
{
    OverlayKey succKey = successorMap.begin()->first;
    NodeHandle succNode = successorMap.begin()->second;
    successorMap.erase(successorMap.find(succKey));

    // ensure that thisNode is always in the successor list
    if (getSize() == 0) {
        addSuccessor(thisNode);
        succNode = thisNode;
    }

    return succNode;
}

void SuccessorList::removeSuccessor(NodeHandle successor)
{
    OverlayKey tmp = successor.key - (thisNode.key + OverlayKey::ONE);
    std::map<OverlayKey, NodeHandle>::iterator iter = successorMap.find(tmp);
    if (iter != successorMap.end()) {
        if (iter->second != successor)
            cout << "wrong mapping" << endl;
        successorMap.erase(iter);
    }

    // ensure that thisNode is always in the successor list
    if (getSize() == 0)
        addSuccessor(thisNode);
}

void SuccessorList::updateDisplayString()
{
// FIXME: doesn't work without tcl/tk
    	if (ev.isGUI()) {
        char buf[80];

        if (successorMap.size() == 1) {
            sprintf(buf, "1 successor");
        } else {
            sprintf(buf, "%zi successors", successorMap.size());
        }

        displayString().setTagArg("t", 0, buf);
        displayString().setTagArg("t", 2, "blue");
    }

}

void SuccessorList::updateTooltip()
{
    if (ev.isGUI()) {
        std::stringstream str;
        for (uint i = 0; i < successorMap.size(); i++)	{
            str << getSuccessor(i);
            if ( i != successorMap.size() - 1 )
                str << endl;
        }


        char buf[1024];
        sprintf(buf, "%s", str.str().c_str());
        displayString().setTagArg("tt", 0, buf);
    }
}

void SuccessorList::display()
{
    cout << "Content of SuccessorList:" << endl;
    for (std::map<OverlayKey,NodeHandle>::iterator it = successorMap.begin(); it != successorMap.end(); it++)
        cout << it->first << " with Node: " << it->second << endl;
}
