//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file SuccessorList.h
 * @author Markus Mauch
 */

#ifndef __SUCCESSORLIST_H_
#define __SUCCESSORLIST_H_

#include <set>
#include <vector>

#include <omnetpp.h>

#include <NodeHandle.h>
#include <InitStages.h>

/**
 * Successor list module
 *
 * This modul contains the successor list of the Chord implementation.
 *
 * @author Markus Mauch, Ingmar Baumgart
 * @see Chord
 */
class SuccessorList : public cSimpleModule
{
  public:
    virtual int numInitStages() const
    {
        return MAX_STAGE_OVERLAY + 1;
    }
    virtual void initialize(int stage);
    virtual void handleMessage(cMessage* msg);

    /**
     * Initializes the successor list. This should be called on startup
     *
     * @param size maximum number of neighbors in the successor list
     * @param owner the node owner is added to the successor list
     */
    virtual void initializeList(uint size, NodeHandle owner);

    /**
     * Returns number of neighbors in the successor list
     *
     * @return number of neighbors
     */
    virtual uint getSize();

    /**
     * Clears the successor list
     */
    virtual void clear();

    /**
     * Checks if the successor list is empty
     *
     * @return returns false if the successor list contains other nodes
     *         than this node, true otherwise.
     */
    virtual bool isEmpty();

    /**
     * Returns a particular successor
     *
     * @param pos position in the successor list
     * @return successor at position pos
     */
    virtual const NodeHandle& getSuccessor(uint pos = 0);

    /**
     * Adds new successor nodes to the successor list
     *
     * Adds new successor nodes to the successor list and sorts the
     * list using the corresponding chord keys. If the list size exceeds
     * the maximum size nodes at the end of the list will be removed.
     *
     * @param successor the node handle of the successor to be added
     */
    virtual void addSuccessor(NodeHandle successor);

    /**
     * Removes all occurrences of the first successor node from the list
     *
     * @return the removed successor node
     */
    virtual NodeHandle popSuccessor();

    /**
     * Removes a certain successor from the successor list
     *
     * @param successor the successor to be removed
     */
    virtual void removeSuccessor(NodeHandle successor);
    void display ();


  protected:
    NodeHandle thisNode; /**< own node handle */
    std::map<OverlayKey, NodeHandle> successorMap; /**< internal representation of the successor list */

    uint successorListSize; /**< maximum size of the successor list */

    /**
     * Derives a new successor list from the internal successor set
     *
     * Derives a new successor list from the internal successor set.
     * This method is called whenever nodes are added or removed
     * from the set.
     */
    void buildSuccessorList();

    /**
     * Displays the current number of successors in the list
     */
    void updateDisplayString();

    /**
     * Displays the first 4 successor nodes as tooltip.
     */
    void updateTooltip();
};

#endif
