//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file Gia.cc 
 * @author Robert Palmer
 */

#include <UDPControlInfo_m.h>
#include <IPAddressResolver.h>

#include <CommonAPIMessages_m.h>
#include <ExtAPIMessages_m.h>
#include <InitStages.h>

#include "Gia.h"


Define_Module(Gia);

void Gia::initializeOverlay(int stage)
{
    // wait until IPAddressResolver initialized all interfaces and assigns addresses
    if(stage != MIN_STAGE_OVERLAY)
        return;

    // Get parameters from omnetpp.ini
    maxNeighbors = par("maxNeighbors");
    minNeighbors = par("minNeighbors");
    maxTopAdaptionInterval = par("maxTopAdaptionInterval");
    topAdaptionAggressiveness = par("topAdaptionAggressiveness");
    maxLevelOfSatisfaction = par("maxLevelOfSatisfaction");
    updateDelay = par("updateDelay");
    maxHopCount = par("maxHopCount"); //obsolete
    messageTimeout = par("messageTimeout");
    neighborTimeout = par("neighborTimeout");
    tokenWaitTime = par("tokenWaitTime");
    keyListDelay = par("keyListDelay");
    outputNodeDetails = par("outputNodeDetails");
    optimizeReversePath = par("optimizeReversePath");

    onlyCommonAPIMessages = false;

    // get references on necessary modules
    keyListModule = check_and_cast<KeyListModule*>
                    (parentModule()->submodule("keyListModule"));
    neighbors = check_and_cast<Neighbors*>
                (parentModule()->submodule("neighbors"));
    tokenFactory = check_and_cast<TokenFactory*>
                   (parentModule()->submodule("tokenFactory"));

    msgBookkeepingList = new MessageBookkeeping(neighbors, messageTimeout);

    // clear neighbor candidate list
    neighCand.clear();
    knownNodes.clear();

    WATCH(thisNode);
    WATCH(bootstrapNode);
    WATCH(levelOfSatisfaction);

    // self-messages
    satisfaction_timer = new cMessage("satisfaction_timer");
    update_timer = new cMessage("update_timer");
    timedoutMessages_timer = new cMessage("timedoutMessages_timer");
    timedoutNeighbors_timer = new cMessage("timedoutNeighbors_timer");
    sendKeyList_timer = new cMessage("sendKeyList_timer");

    // statistics
    stat_joinCount = 0;
    stat_joinBytesSent = 0;
    stat_joinREQ = 0;
    stat_joinREQBytesSent = 0;
    stat_joinRSP = 0;
    stat_joinRSPBytesSent = 0;
    stat_joinACK = 0;
    stat_joinACKBytesSent = 0;
    stat_joinDNY = 0;
    stat_joinDNYBytesSent = 0;
    stat_disconnectMessages = 0;
    stat_disconnectMessagesBytesSent = 0;
    stat_updateMessages = 0;
    stat_updateMessagesBytesSent = 0;
    stat_tokenMessages = 0;
    stat_tokenMessagesBytesSent = 0;
    stat_keyListMessages = 0;
    stat_keyListMessagesBytesSent = 0;
    stat_routeMessages = 0;
    stat_routeMessagesBytesSent = 0;
    stat_maxNeighbors = 0;
    stat_addedNeighbors = 0;
    stat_removedNeighbors = 0;
    stat_numSatisfactionMessages = 0;
    stat_sumLevelOfSatisfaction = 0.0;
    stat_maxLevelOfSatisfaction = 0.0;

    changeState(INIT);
    if(bootstrapNode.isUnspecified())
        changeState(OVERLAY_NOT_READY);
}

void Gia::changeState(int toState)
{
    switch (toState) {
    case INIT: {
            state = INIT;

            // set up local nodehandle
            NodeHandle thisNodeHandle;
            thisNodeHandle.ip = IPAddressResolver().
                                addressOf(parentModule()->parentModule()).get4();
            thisNodeHandle.key = OverlayKey::random();
            thisNodeHandle.port = localPort;
            thisNodeHandle.moduleId = parentModule()->id();

            thisNode.setNodeHandle(thisNodeHandle);

            // get possible bandwidth from ppp-Module
            double capacity = 0;

            cModule* nodeModule = parentModule()->parentModule();

            int gateSize = nodeModule->gateSize("in");

            if(gateSize == 0)
                capacity += uniform(1,800000);
            else {
                // this relies on IPv4Underlay
                for (int i=0; i<gateSize; i++) {
                    int gateID = nodeModule->gate("in",0)->id()+i;
                    cGate* currentGate = nodeModule->gate(gateID);
                    if (currentGate->isConnected())
                        capacity += check_and_cast<cBasicChannel *>
                                    (currentGate->fromGate()->channel())->datarate()
                                    - uniform(0,800000);
                }
            }

            thisNode.setCapacity(capacity);
            thisNode.setConnectionDegree(0);
            thisNode.setReceivedTokens(0);
            thisNode.setSentTokens(0);
            if (outputNodeDetails)
                EV << "(Gia) Node details: " << thisNode << endl;

            // get our entry point to GIA-Overlay and register
            // our own node at BootstrapOracle
            bootstrapNode = bootstrapOracle->getBootstrapNode();
            if(!(bootstrapNode.isUnspecified()))
                knownNodes.add(bootstrapNode);
            else {
                assert(!(thisNode.getNodeHandle().isUnspecified()));
                bootstrapOracle->registerPeer(thisNode.getNodeHandle());
            }

            // register node at TokenFactory
            //tokenFactory->setNode(&thisNode);
            tokenFactory->setNeighbors(neighbors);
            tokenFactory->setMaxHopCount(maxHopCount);

            if (debugOutput)
                EV << "(Gia) Node " << thisNode.getNodeHandle().key
                << " (" << thisNode.getNodeHandle().ip << ":"
                << thisNode.getNodeHandle().port << ") with capacity: "
                << thisNode.getCapacity()  << " entered INIT state." << endl;

            parentModule()->parentModule()->bubble("Enter INIT state.");

            // schedule satisfaction_timer
            cancelEvent(satisfaction_timer);
            scheduleAt(simulation.simTime(), satisfaction_timer);

            // schedule timedoutMessages_timer
            scheduleAt(simulation.simTime() + messageTimeout,
                       timedoutMessages_timer);
            scheduleAt(simulation.simTime() + neighborTimeout,
                       timedoutNeighbors_timer);

            break;
        }
    case OVERLAY_READY:
        state = OVERLAY_READY;
        break;

    case OVERLAY_NOT_READY:
        state = OVERLAY_NOT_READY;
        break;

    }
    setBootstrapedIcon();
}

void Gia::setBootstrapedIcon()
{
    if (ev.isGUI()) {
        if (state == OVERLAY_READY) {
            parentModule()->parentModule()->displayString().
            setTagArg("i2", 1, "");
            displayString().setTagArg("i", 1, "");
        } else {
            parentModule()->parentModule()->displayString().
            setTagArg("i2", 1, "red");
            displayString().setTagArg("i", 1, "red");
        }
    }
}

void Gia::handleTimerEvent(cMessage* msg)
{
    if (msg->isName("satisfaction_timer")) {
        // calculate level of satisfaction and select new neighbor candidate

        levelOfSatisfaction = calculateLevelOfSatisfaction();
        stat_numSatisfactionMessages++;
        stat_sumLevelOfSatisfaction += levelOfSatisfaction;
        if (levelOfSatisfaction > stat_maxLevelOfSatisfaction)
            stat_maxLevelOfSatisfaction = levelOfSatisfaction;

        // start again satisfaction_timer
        scheduleAt(simulation.simTime() + (maxTopAdaptionInterval *
                                           pow(topAdaptionAggressiveness,
                                               -(1 - levelOfSatisfaction))),
                   satisfaction_timer);

        // Only search a new neighbor if level of satisfaction is
        // under level of maxLevelOfSatisfaction
        if(levelOfSatisfaction < maxLevelOfSatisfaction) {
            if(knownNodes.getSize() == 0)
                if(neighbors->getSize() == 0 && neighCand.getSize() == 0)
                    knownNodes.add(bootstrapOracle->getBootstrapNode());
                else
                    return;

            NodeHandle possibleNeighbor = knownNodes.getRandomCandidate();

            if(!(possibleNeighbor.isUnspecified()) &&
                    thisNode.getNodeHandle() != possibleNeighbor &&
                    !neighbors->contains(GiaNode(possibleNeighbor)) &&
                    !neighCand.contains(possibleNeighbor)) {
                // try to add new neighbor
                neighCand.add(possibleNeighbor);
                sendMessage_JOIN_REQ(thisNode, possibleNeighbor);
            }
        }
    } else if (msg->isName("update_timer")) {
        // send current capacity and degree to all neighbors
        for (uint i=0; i<neighbors->getSize(); i++) {
            sendMessage_UPDATE(thisNode, neighbors->get
                               (i)->getNodeHandle());
        }
    } else if (msg->isName("timedoutMessages_timer")) {
        // remove timedout messages
        msgBookkeepingList->removeTimedoutMessages();
        scheduleAt(simulation.simTime() + messageTimeout,
                   timedoutMessages_timer);
    } else if (msg->isName("timedoutNeighbors_timer")) {
        // remove timedout neighbors
        neighbors->removeTimedoutNodes();
        scheduleAt(simulation.simTime() + neighborTimeout,
                   timedoutNeighbors_timer);
    } else if (msg->isName("sendKeyList_timer")) {
        if (keyList.getSize() > 0) {
            // send keyList to all of our neighbors
            for (uint i=0; i<neighbors->getSize(); i++)
                sendKeyListToNeighbor(*(neighbors->get
                                        (i)));
        }
    } else {
        // other self-messages are notoken-self-messages
        // with an encapsulated message
        const std::string id = msg->name();
        if (id.substr(0, 16) == std::string("wait-for-token: ")) {
            cMessage* decapsulatedMessage = msg->decapsulate();
            if (dynamic_cast<GiaIDMessage*>(decapsulatedMessage) != NULL) {
                GiaIDMessage* message = check_and_cast<GiaIDMessage*>
                                        (decapsulatedMessage);
                forwardMessage(message, false);
            }
        } else if (id.substr(0, 24) == std::string("wait-for-token-fromapp: ")) {
            cMessage* decapsulatedMessage = msg->decapsulate();
            if (dynamic_cast<GiaIDMessage*>(decapsulatedMessage) != NULL) {
                GiaIDMessage* message = check_and_cast<GiaIDMessage*>
                                        (decapsulatedMessage);
                forwardMessage(message, true);
            }
        }
        delete msg;
    }
}

void Gia::handleUDPMessage(BaseOverlayMessage* msg)
{
    if(debugOutput)
        EV << "(Gia) " << thisNode << " received udp message" << endl;

    cPolymorphic* ctrlInfo = msg->removeControlInfo();
    if(ctrlInfo != NULL)
        delete ctrlInfo;

    // Parse TokenMessages
    if (dynamic_cast<TokenMessage*>(msg) != NULL) {
        TokenMessage* giaMsg = check_and_cast<TokenMessage*>(msg);
        GiaNode oppositeNode(giaMsg->getSrcNode(), giaMsg->getSrcCapacity(),
                             giaMsg->getSrcDegree());

        // Process TOKEN-Message
        if ((giaMsg->getCommand() == TOKEN)
                /* && (giaMsg->getServiceType() == INDICATION)*/) {
            if(debugOutput)
                EV << "(Gia) Received Tokenmessage from "
                << oppositeNode << endl;

            GiaNode* target = neighbors->get
                              (oppositeNode);
            if(!(target->isUnspecified())) {
                // Update neighbor-node
                //target->setSentTokens(target->getSentTokens()); // FIXME?
                target->setReceivedTokens(giaMsg->getDstTokenNr());
                updateNeighborList(giaMsg);
            }
        }
        delete msg;
    }

    // Process Route messages
    else if (dynamic_cast<GiaRouteMessage*>(msg) != NULL) {
        GiaRouteMessage* giaMsg = check_and_cast<GiaRouteMessage*>(msg);
        GiaNode oppositeNode(giaMsg->getSrcNode(), giaMsg->getSrcCapacity(),
                             giaMsg->getSrcDegree());

        if((giaMsg->getCommand() == ROUTE)) {
            if(debugOutput)
                EV << "(Gia) Received ROUTE::IND from " << oppositeNode << endl;

            if(state == OVERLAY_READY) {
                GiaNode* target = neighbors->get
                                  (oppositeNode);
                if(!(target->isUnspecified())) {
                    // Update neighbor-node
                    target->setSentTokens(target->getSentTokens()-1);
                    updateNeighborList(giaMsg);
                    forwardMessage(giaMsg, false);
                } else
                    delete giaMsg; //debug?
            }
        }
    }

    // Process KeyList-Messages
    else if (dynamic_cast<KeyListMessage*>(msg) != NULL) {
        KeyListMessage* giaMsg = check_and_cast<KeyListMessage*>(msg);
        GiaNode oppositeNode(giaMsg->getSrcNode(), giaMsg->getSrcCapacity(),
                             giaMsg->getSrcDegree());

        if (giaMsg->getCommand() == KEYLIST) {
            if (debugOutput)
                EV << "(Gia) " << thisNode
                << " received KEYLIST:IND message" << endl;
            int position = neighbors->getPosition(oppositeNode);
            if (position != -1) {
                // update KeyList in neighborList
                uint keyListSize = giaMsg->getKeysArraySize();
                KeyList neighborKeyList;
                for (uint k = 0; k < keyListSize; k++)
                    neighborKeyList.addKeyItem(giaMsg->getKeys(k));
                neighbors->setNeighborKeyList(position, neighborKeyList);
            }
        }
        delete giaMsg;
    }

    // Process Search-Messages
    else if (dynamic_cast<SearchMessage*>(msg) != NULL) {
        SearchMessage* giaMsg = check_and_cast<SearchMessage*>(msg);
        GiaNode oppositeNode(giaMsg->getSrcNode(), giaMsg->getSrcCapacity(),
                             giaMsg->getSrcDegree());

        GiaNode* target = neighbors->get
                          (oppositeNode);

        if(!(target->isUnspecified())) {
            target->setSentTokens(target->getSentTokens()-1);
            updateNeighborList(giaMsg);
            processSearchMessage(giaMsg, false);
        } else {
            EV << "(Gia) Message " << msg << " dropped!" << endl;
            RECORD_STATS(numDropped++; bytesDropped += msg->byteLength());
            delete msg;
        }
    }

    // Process Search-Response-Messages
    else if (dynamic_cast<SearchResponseMessage*>(msg) != NULL) {
        SearchResponseMessage* responseMsg =
            check_and_cast<SearchResponseMessage*>(msg);
        forwardSearchResponseMessage(responseMsg);
    }

    // Process Gia messages
    else if (dynamic_cast<GiaMessage*>(msg) != NULL) {
        GiaMessage* giaMsg = check_and_cast<GiaMessage*>(msg);

        assert(giaMsg->getSrcNode().moduleId != -1);
        GiaNode oppositeNode(giaMsg->getSrcNode(), giaMsg->getSrcCapacity(),
                             giaMsg->getSrcDegree());

        if (debugOutput)
            EV << "(Gia) " << thisNode << " received GIA- message from "
            << oppositeNode << endl;
        updateNeighborList(giaMsg);

        // Process JOIN:REQ messages
        if (giaMsg->getCommand() == JOIN_REQUEST) {
            if (debugOutput)
                EV << "(Gia) " << thisNode
                << " received JOIN:REQ message" << endl;
            if (acceptNode(oppositeNode)) {
                neighCand.add(oppositeNode.getNodeHandle());
                sendMessage_JOIN_RSP(thisNode, oppositeNode.getNodeHandle());
            } else {
                if (debugOutput)
                    EV << "(Gia) " << thisNode << " denies node "
                    << oppositeNode << endl;
                sendMessage_JOIN_DNY(thisNode, oppositeNode.getNodeHandle());
            }
        }

        // Process JOIN:RSP messages
        else if (giaMsg->getCommand() == JOIN_RESPONSE) {
            if(debugOutput)
                EV << "(Gia) " << thisNode << " received JOIN:RSP message"
                << endl;
            if(neighCand.contains(oppositeNode.getNodeHandle())) {
                neighCand.remove(oppositeNode.getNodeHandle());
                if(acceptNode(oppositeNode)) {
                    addNeighbor(oppositeNode);

                    GiaNeighborMessage* msg =
                        check_and_cast<GiaNeighborMessage*>(giaMsg);
                    for(uint i = 0; i < msg->getNeighborsArraySize(); i++) {
                        GiaNode temp = msg->getNeighbors(i);
                        if(temp != thisNode && temp != oppositeNode)
                            knownNodes.add(temp.getNodeHandle());
                    }

                    sendMessage_JOIN_ACK(thisNode,oppositeNode.getNodeHandle());
                } else {
                    if (debugOutput)
                        EV << "(Gia) " << thisNode << " denies node "
                        << oppositeNode << endl;
                    sendMessage_JOIN_DNY(thisNode,oppositeNode.getNodeHandle());
                }
            }
        }

        // Process JOIN:ACK messages
        else if (giaMsg->getCommand() == JOIN_ACK) {
            if (debugOutput)
                EV << "(Gia) " << thisNode << " received JOIN:ACK message"
                << endl;
            if (neighCand.contains(oppositeNode.getNodeHandle()) &&
                    neighbors->getSize() < maxNeighbors) {
                neighCand.remove(oppositeNode.getNodeHandle());
                addNeighbor(oppositeNode);

                GiaNeighborMessage* msg =
                    check_and_cast<GiaNeighborMessage*>(giaMsg);

                for(uint i = 0; i < msg->getNeighborsArraySize(); i++) {
                    GiaNode temp = msg->getNeighbors(i);
                    if(temp != thisNode && temp != oppositeNode)
                        knownNodes.add(msg->getNeighbors(i).getNodeHandle());
                }
            }
        }

        // Process JOIN:DNY messages
        else if (giaMsg->getCommand() == JOIN_DENY) {
            if (debugOutput)
                EV << "(Gia) " << thisNode << " received JOIN:DNY message"
                << endl;

            if (neighCand.contains(oppositeNode.getNodeHandle()))
                neighCand.remove(oppositeNode.getNodeHandle());
            knownNodes.remove(oppositeNode.getNodeHandle());

        }


        // Process DISCONNECT-Message
        else if (giaMsg->getCommand() == DISCONNECT) {
            if (debugOutput)
                EV << "(Gia) " << thisNode << " received DISCONNECT:IND message" << endl;
            if (neighbors->contains(oppositeNode))
                removeNeighbor(oppositeNode);
        }

        // Process UPDATE-Message
        else if (giaMsg->getCommand() == UPDATE) {
            if (debugOutput)
                EV << "(Gia) " << thisNode << " received UPDATE:IND message"
                << endl;

            if (neighbors->contains(oppositeNode)) {
                // update ConnectionDegree and Capacity
                GiaNode* target = neighbors->get
                                  (oppositeNode);
                if(!(target->isUnspecified())) {
                    target->setConnectionDegree(
                        oppositeNode.getConnectionDegree());

                    target->setCapacity(oppositeNode.getCapacity());
                }
            }
        } else {
            // Show unknown gia-messages
            if (debugOutput) {
                EV << "(Gia) NODE: "<< thisNode << endl
                << "       Command: " << giaMsg->getCommand() << endl
                << "       HopCount: " << giaMsg->getHopCount() << endl
                << "       SrcKey: " << giaMsg->getSrcNode().key << endl
                << "       SrcIP: " << giaMsg->getSrcNode().key << endl
                << "       SrcPort: " << giaMsg->getSrcNode().port << endl
                << "       SrcModuleID: " << giaMsg->getSrcNode().moduleId
                << endl
                << "       SrcCapacity: " << giaMsg->getSrcCapacity() << endl
                << "       SrcDegree: " << giaMsg->getSrcDegree() << endl;

                RECORD_STATS(numDropped++;bytesDropped += giaMsg->byteLength());
            }
        }
        delete giaMsg;
    } else // PROCESS other messages than GiaMessages
        delete msg; // delete them all
}

bool Gia::acceptNode(const GiaNode& nNode)
{
    if (neighbors->contains(nNode))
        return false;

    if (neighbors->getSize() < maxNeighbors) {
        // we have room for new node: accept node
        return true;
    }
    // we need to drop a neighbor
    else {
        // determine node with highest capacity
        std::vector<GiaNode> dropCandidateList;
        double maxCapacity = 0;
        GiaNode dropCandidate;
        for (uint i=0; i<neighbors->getSize(); i++) {
            if (neighbors->get
                    (i)->getCapacity() <= nNode.getCapacity()) {
                dropCandidateList.push_back(*(neighbors->get
                                              (i)));
                if (neighbors->get
                        (i)->getCapacity() > maxCapacity) {
                    dropCandidate = *(neighbors->get
                                      (i));
                    maxCapacity = dropCandidate.getCapacity();
                }
            }
        }

        // has nNode lower capacity than all of our neighbors?
        if (dropCandidateList.size() == 0)
            return false;
        else {
            if((dropCandidateList.size() == neighbors->getSize()
                    || dropCandidate.getConnectionDegree() >
                    nNode.getConnectionDegree())
                    && dropCandidate.getConnectionDegree() > 1) {
                if (debugOutput)
                    EV << "(Gia) " << thisNode << " drops "
                    << dropCandidate << " in favor of " << nNode << endl;
                removeNeighbor(dropCandidate);
                sendMessage_DISCONNECT(thisNode, dropCandidate.getNodeHandle());
                return true;
            } else
                return false;
        }
    }
    return false;
}

void Gia::addNeighbor(GiaNode& node)
{
    assert(neighbors->getSize() < maxNeighbors);

    stat_addedNeighbors++;
    EV << "(Gia) " << thisNode << " accepted new neighbor " << node << endl;
    parentModule()->parentModule()->bubble("New neighbor");
    thisNode.setConnectionDegree(neighbors->getSize());
    changeState(OVERLAY_READY);
    if (stat_maxNeighbors < neighbors->getSize()) {
        stat_maxNeighbors = neighbors->getSize();
    }

    cancelEvent(update_timer);
    scheduleAt(simulation.simTime() + updateDelay, update_timer);

    node.setSentTokens(5);
    node.setReceivedTokens(5);

    // send keyList to new Neighbor
    if (keyList.getSize() > 0)
        sendKeyListToNeighbor(node);
    neighbors->add
    (node);

    showOverlayNeighborArrow(node.getNodeHandle(), false,
                             "m=m,50,0,50,0;o=red,1");
    bootstrapOracle->registerPeer(thisNode.getNodeHandle());
}

void Gia::removeNeighbor(const GiaNode& node)
{
    stat_removedNeighbors++;
    int position = neighbors->getPosition(node);
    if (position != -1) {
        if (debugOutput)
            EV << "(Gia) " << thisNode << " removes " << node
            << " from neighborlist." << endl;
        neighbors->remove
        (position);

        thisNode.setConnectionDegree(neighbors->getSize());
        if (neighbors->getSize() == 0)
            changeState(OVERLAY_NOT_READY);
    }
    deleteOverlayNeighborArrow(node.getNodeHandle());

    cancelEvent(update_timer);
    scheduleAt(simulation.simTime() + updateDelay, update_timer);
}

double Gia::calculateLevelOfSatisfaction()
{
    uint neighborsCount = neighbors->getSize();
    if(neighborsCount < minNeighbors)
        return 0.0;

    double total = 0.0;
    double levelOfSatisfaction = 0.0;

    for (uint i=0; i < neighborsCount; i++)
        total += neighbors->get
                 (i)->getCapacity() / neighborsCount;

    assert(thisNode.getCapacity() != 0);
    levelOfSatisfaction = total / thisNode.getCapacity();

    if ((levelOfSatisfaction > 1.0) || (neighborsCount >= maxNeighbors))
        return 1.0;
    return levelOfSatisfaction;
}


void Gia::sendMessage_JOIN_REQ(const GiaNode& src, const NodeHandle& dst)
{
    // send JOIN:REQ message
    GiaMessage* msg = new GiaMessage("JOIN_REQ");
    msg->setCommand(JOIN_REQUEST);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());
    msg->setLength(GIA_L(msg));

    stat_joinCount += 1;
    stat_joinBytesSent += msg->byteLength();
    stat_joinREQ += 1;
    stat_joinREQBytesSent += msg->byteLength();

    sendMessageToUDP(dst, msg);
}

void Gia::sendMessage_JOIN_RSP(const GiaNode& src, const NodeHandle& dst)
{
    // send JOIN:RSP message
    GiaNeighborMessage* msg = new GiaNeighborMessage("JOIN_RSP");
    msg->setCommand(JOIN_RESPONSE);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());

    msg->setNeighborsArraySize(neighbors->getSize());
    //TODO: add parameter maxSendNeighbors
    for(uint i = 0; i < neighbors->getSize(); i++)
        msg->setNeighbors(i, *(neighbors->get
                               (i)));

    msg->setLength(GIANEIGHBOR_L(msg));

    stat_joinCount += 1;
    stat_joinBytesSent += msg->byteLength();
    stat_joinRSP += 1;
    stat_joinRSPBytesSent += msg->byteLength();

    sendMessageToUDP(dst, msg);
}

void Gia::sendMessage_JOIN_ACK(const GiaNode& src, const NodeHandle& dst)
{
    // send JOIN:ACK message
    GiaNeighborMessage* msg = new GiaNeighborMessage("JOIN_ACK");
    msg->setCommand(JOIN_ACK);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());

    msg->setNeighborsArraySize(neighbors->getSize());
    //TODO: add parameter maxSendNeighbors
    for(uint i = 0; i < neighbors->getSize(); i++)
        msg->setNeighbors(i, *(neighbors->get
                               (i)));

    msg->setLength(GIANEIGHBOR_L(msg));

    stat_joinCount += 1;
    stat_joinBytesSent += msg->byteLength();
    stat_joinACK += 1;
    stat_joinACKBytesSent += msg->byteLength();

    sendMessageToUDP(dst, msg);
}

void Gia::sendMessage_JOIN_DNY(const GiaNode& src, const NodeHandle& dst)
{
    // send JOIN:DNY message
    GiaMessage* msg = new GiaMessage("JOIN_DENY");
    msg->setCommand(JOIN_DENY);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());
    msg->setLength(GIA_L(msg));

    stat_joinCount += 1;
    stat_joinBytesSent += msg->byteLength();
    stat_joinDNY += 1;
    stat_joinDNYBytesSent += msg->byteLength();

    sendMessageToUDP(dst, msg);
}

void Gia::sendMessage_DISCONNECT(const GiaNode& src, const NodeHandle& dst)
{
    // send DISCONNECT:IND message
    GiaMessage* msg = new GiaMessage("DISCONNECT");
    msg->setCommand(DISCONNECT);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());
    msg->setLength(GIA_L(msg));

    stat_disconnectMessagesBytesSent += msg->byteLength();
    stat_disconnectMessages += 1;

    sendMessageToUDP(dst, msg);
}

void Gia::sendMessage_UPDATE(const GiaNode& src, const NodeHandle& dst)
{
    // send UPDATE:IND message
    GiaMessage* msg = new GiaMessage("UPDATE");
    msg->setCommand(UPDATE);
    msg->setSrcNode(src.getNodeHandle());
    msg->setSrcCapacity(src.getCapacity());
    msg->setSrcDegree(src.getConnectionDegree());
    msg->setLength(GIA_L(msg));

    stat_updateMessagesBytesSent += msg->byteLength();
    stat_updateMessages += 1;

    sendMessageToUDP(dst, msg);
}

void Gia::sendKeyListToNeighbor(const GiaNode& dst)
{
    // send KEYLIST:IND message
    KeyListMessage* msg = new KeyListMessage("KEYLIST");
    msg->setCommand(KEYLIST);
    msg->setSrcNode(thisNode.getNodeHandle());
    msg->setSrcCapacity(thisNode.getCapacity());
    msg->setSrcDegree(thisNode.getConnectionDegree());

    msg->setKeysArraySize(keyList.getSize());
    for (uint i=0; i<keyList.getSize(); i++)
        msg->setKeys(i, keyList.get(i));

    msg->setLength(KEYLIST_L(msg));

    stat_keyListMessagesBytesSent += msg->byteLength();
    stat_keyListMessages += 1;

    sendMessageToUDP(dst.getNodeHandle(), msg);
}

void Gia::sendToken(const GiaNode& dst)
{
    //GiaNode node = tokenQueue.top();
    TokenMessage* tokenMsg = new TokenMessage("TOKEN");
    tokenMsg->setCommand(TOKEN);
    tokenMsg->setHopCount(maxHopCount);
    tokenMsg->setSrcNode(thisNode.getNodeHandle());
    tokenMsg->setSrcCapacity(thisNode.getCapacity());
    tokenMsg->setSrcDegree(thisNode.getConnectionDegree());
    tokenMsg->setSrcTokenNr(dst.getReceivedTokens());
    tokenMsg->setDstTokenNr(dst.getSentTokens()+1);
    tokenMsg->setLength(TOKEN_L(tokenMsg));

    stat_tokenMessagesBytesSent += tokenMsg->byteLength();
    stat_tokenMessages += 1;

    sendMessageToUDP(dst.getNodeHandle(), tokenMsg);
}

void Gia::updateNeighborList(GiaMessage* msg)
{
    if(neighbors->contains(msg->getSrcNode().key)) {
        GiaNode* target = neighbors->get
                          (msg->getSrcNode().key);
        target->setConnectionDegree(msg->getSrcDegree());
        target->setCapacity(msg->getSrcCapacity());
        neighbors->updateTimestamp(*target);
    }
}

void Gia::forwardSearchResponseMessage(SearchResponseMessage* responseMsg)
{
    if (responseMsg->getHopCount() != 0) {
        uint reversePathArraySize = responseMsg->getReversePathArraySize();

        // reached node which started search?
        if (reversePathArraySize == 0) {
            deliverSearchResult(responseMsg);
        }
        // forward message to next node in reverse path list
        else {
            GiaNode* targetNode = neighbors->get
                                  (responseMsg->getReversePath(
                                       reversePathArraySize-1));

            if(!(targetNode->isUnspecified())) {
                responseMsg->setDestKey(targetNode->getNodeHandle().key);
                responseMsg->setReversePathArraySize(reversePathArraySize-1);
                for (uint i=0; i<reversePathArraySize-1; i++)
                    responseMsg->setReversePath(i,
                                                responseMsg->getReversePath(i));
                responseMsg->setLength(responseMsg->length() - KEY_L);

                stat_routeMessagesBytesSent += responseMsg->byteLength();
                stat_routeMessages += 1;

                if(responseMsg->getHopCount() > 0)
                    RECORD_STATS(numForwarded++; bytesForwarded +=
                                     responseMsg->byteLength());

                sendMessageToUDP(targetNode->getNodeHandle(), responseMsg);
            } else {
                EV << "(Gia) wrong reverse path in " << *responseMsg
                << " ... deleted!" << endl;
                RECORD_STATS(numDropped++;
                             bytesDropped += responseMsg->byteLength());
                delete responseMsg;
            }
        }
    }
    // max hopcount reached. delete message
    else
        delete responseMsg;
}

void Gia::forwardMessage(GiaIDMessage* msg , bool fromApplication)
{
    if (msg->getHopCount() == 0) {
        // Max-Hopcount reached. Discard message
        if (!fromApplication)
            tokenFactory->grantToken();
        RECORD_STATS(numDropped++; bytesDropped += msg->byteLength());
        delete msg;
    } else {
        // local delivery?
        if (msg->getDestKey() == thisNode.getNodeHandle().key) {
            if(!fromApplication)
                tokenFactory->grantToken();

            if(debugOutput)
                EV << "(Gia) Deliver messsage " << msg
                << " to application at " << thisNode << endl;

            OverlayCtrlInfo* overlayCtrlInfo =
                new OverlayCtrlInfo();

            overlayCtrlInfo->setThisNode(thisNode.getNodeHandle());
            overlayCtrlInfo->setHopCount(msg->getHopCount());
            overlayCtrlInfo->setDestKey(msg->getDestKey());
            overlayCtrlInfo->setSrcNode(msg->getSrcNode());

            msg->setControlInfo(overlayCtrlInfo);
            callDeliver(msg);
        } else {
            // forward message to next hop

            // do we know the destination-node?
            if (neighbors->contains(msg->getDestKey())) {
                // yes, destination-Node is our neighbor
                GiaNode* target = neighbors->get
                                  (msg->getDestKey());

                if (target->getReceivedTokens() == 0) {
                    // wait for free token
                    if (debugOutput)
                        EV << "(Gia) No free Node, wait for free token" << endl;

                    //bad code
                    std::string id;
                    if (!fromApplication)
                        id = "wait-for-token: " + msg->getID().toString();
                    else
                        id = "wait-for-token-fromapp: " +
                             msg->getID().toString();
                    cMessage* wait_timer = new cMessage(id.c_str());
                    wait_timer->encapsulate(msg);
                    scheduleAt(simulation.simTime() + tokenWaitTime,wait_timer);
                    return;
                } else {
                    // decrease nextHop-tokencount
                    target->setReceivedTokens(target->getReceivedTokens()-1);

                    // forward msg to nextHop
                    msg->setHopCount(msg->getHopCount()-1);
                    msg->setSrcNode(thisNode.getNodeHandle());
                    msg->setSrcCapacity(thisNode.getCapacity());
                    msg->setSrcDegree(thisNode.getConnectionDegree());
                    if (debugOutput)
                        EV << "(Gia) Forwarding message " << msg
                        << " to neighbor " << *target << endl;
                    if (!fromApplication)
                        tokenFactory->grantToken();

                    stat_routeMessagesBytesSent += msg->byteLength();
                    stat_routeMessages += 1;

                    if(msg->getHopCount() > 0)
                        RECORD_STATS(numForwarded++; bytesForwarded +=
                                         msg->byteLength());

                    sendMessageToUDP(target->getNodeHandle(), msg);
                }
            } else {
                // no => pick node with at least one token and highest capacity
                if (!msgBookkeepingList->contains(msg))
                    msgBookkeepingList->addMessage(msg);
                GiaNode* nextHop = msgBookkeepingList->getNextHop(msg);

                // do we have a neighbor who send us a token?
                if (nextHop == NULL) {
                    // wait for free token
                    if (debugOutput)
                        EV << "(Gia) No free Node, wait for free token" << endl;
                    std::string id;
                    if (!fromApplication)
                        id = "wait-for-token: " + msg->getID().toString();
                    else
                        id = "wait-for-token-fromapp: " +
                             msg->getID().toString();
                    cMessage* wait_timer = new cMessage(id.c_str());
                    wait_timer->encapsulate(msg);
                    scheduleAt(simulation.simTime() + tokenWaitTime,wait_timer);
                    return;
                } else {
                    // decrease nextHop-tokencount
                    nextHop->setReceivedTokens(nextHop->getReceivedTokens()-1);

                    // forward msg to nextHop
                    msg->setHopCount(msg->getHopCount()-1);
                    msg->setSrcNode(thisNode.getNodeHandle());
                    msg->setSrcCapacity(thisNode.getCapacity());
                    msg->setSrcDegree(thisNode.getConnectionDegree());
                    if (debugOutput)
                        EV << "(Gia) Forwarding message " << msg
                        << " to " << *nextHop << endl;
                    if (!fromApplication)
                        tokenFactory->grantToken();

                    stat_routeMessagesBytesSent += msg->byteLength();
                    stat_routeMessages += 1;

                    if(msg->getHopCount() > 0)
                        RECORD_STATS(numForwarded++; bytesForwarded +=
                                         msg->byteLength());

                    sendMessageToUDP(nextHop->getNodeHandle(), msg);
                }
            }
        }
    }
}

void Gia::route(const OverlayKey& key, cMessage* msg, const NodeHandle& hint)
{
    if (state == OVERLAY_READY) {
        GiaRouteMessage* routeMsg = new GiaRouteMessage("ROUTE");
        routeMsg->setSignaling(false);
        routeMsg->setCommand(ROUTE);
        routeMsg->setHopCount(maxHopCount);
        routeMsg->setDestKey(key);
        routeMsg->setSrcNode(thisNode.getNodeHandle());
        routeMsg->setSrcCapacity(thisNode.getCapacity());
        routeMsg->setSrcDegree(thisNode.getConnectionDegree());
        routeMsg->setOriginatorKey(thisNode.getNodeHandle().key);
        routeMsg->setOriginatorIP(thisNode.getNodeHandle().ip);
        routeMsg->setOriginatorPort(thisNode.getNodeHandle().port);
        routeMsg->setID(OverlayKey::random());
        routeMsg->encapsulate(msg);
        routeMsg->setLength(GIAROUTE_L(routeMsg));

        forwardMessage(routeMsg, true);
    } else {
        RECORD_STATS(numDropped++; bytesDropped += msg->byteLength());
        delete msg;
    }
}

void Gia::handleAppMessage(cMessage* msg)
{
    // do we receive a keylist?
    if (dynamic_cast<GIAput*>(msg) != NULL) {
        GIAput* putMsg = check_and_cast<GIAput*>(msg);
        uint keyListSize = putMsg->getKeysArraySize();
        for (uint k=0; k<keyListSize; k++)
            keyList.addKeyItem(putMsg->getKeys(k));

        // actualize vector in KeyListModule
        keyListModule->setKeyListVector(keyList.getVector());

        scheduleAt(simulation.simTime() + keyListDelay, sendKeyList_timer);

        delete putMsg;
    } else if (dynamic_cast<GIAsearch*>(msg) != NULL) {
        if (state == OVERLAY_READY) {
            GIAsearch* getMsg = check_and_cast<GIAsearch*>(msg);
            SearchMessage* searchMsg = new SearchMessage("SEARCH");
            searchMsg->setCommand(SEARCH);
            searchMsg->setSignaling(false);
            searchMsg->setHopCount(maxHopCount);
            searchMsg->setDestKey(getMsg->getSearchKey());
            searchMsg->setSrcNode(thisNode.getNodeHandle());
            searchMsg->setSrcCapacity(thisNode.getCapacity());
            searchMsg->setSrcDegree(thisNode.getConnectionDegree());
            searchMsg->setSearchKey(getMsg->getSearchKey());
            searchMsg->setMaxResponses(getMsg->getMaxResponses());
            searchMsg->setReversePathArraySize(0);
            searchMsg->setID(OverlayKey::random());
            searchMsg->setLength(SEARCH_L(searchMsg));

            processSearchMessage(searchMsg, true);

            delete getMsg;
        } else
            delete msg;
    } else {
        delete msg;
        EV << "(Gia) unkown message from app deleted!" << endl;
    }
}


void Gia::sendSearchResponseMessage(const GiaNode& srcNode, SearchMessage* msg)
{
    // does SearchMessage->foundNode[] already contain this node
    uint foundNodeArraySize = msg->getFoundNodeArraySize();
    bool containsNode = false;
    for (uint i=0; i<foundNodeArraySize; i++)
        if (srcNode.getNodeHandle().key == msg->getFoundNode(i))
            containsNode = true;

    if (!containsNode && msg->getMaxResponses() > 0) {
        // add this node to SearchMessage->foundNode[]
        msg->setFoundNodeArraySize(foundNodeArraySize+1);
        msg->setFoundNode(foundNodeArraySize, srcNode.getNodeHandle().key);

        // decrease SearchMessage->maxResponses
        msg->setMaxResponses(msg->getMaxResponses()-1);

        // get first node in reverse-path
        uint reversePathArraySize = msg->getReversePathArraySize();

        if (reversePathArraySize == 0) {
            // we have the key
            // deliver response to application
            SearchResponseMessage* responseMsg =
                new SearchResponseMessage("ANSWER");
            responseMsg->setCommand(ANSWER);
            responseMsg->setSignaling(false);
            responseMsg->setHopCount(maxHopCount);
            responseMsg->setSrcNode(thisNode.getNodeHandle());
            responseMsg->setSrcCapacity(thisNode.getCapacity());
            responseMsg->setSrcDegree(thisNode.getConnectionDegree());
            responseMsg->setSearchKey(msg->getSearchKey());
            responseMsg->setFoundNode(srcNode);
            responseMsg->setID(OverlayKey::random());
            responseMsg->setSearchHopCount(0);

            responseMsg->setLength(SEARCHRESPONSE_L(responseMsg));

            deliverSearchResult(responseMsg);
        } else {
            uint reversePathArraySize(msg->getReversePathArraySize());
            SearchResponseMessage* responseMsg =
                new SearchResponseMessage("ANSWER");
            responseMsg->setCommand(ANSWER);
            responseMsg->setHopCount(maxHopCount);
            responseMsg->setSrcNode(srcNode.getNodeHandle());
            responseMsg->setSrcCapacity(srcNode.getCapacity());
            responseMsg->setSrcDegree(srcNode.getConnectionDegree());
            responseMsg->setSearchKey(msg->getSearchKey());
            responseMsg->setFoundNode(srcNode);
            responseMsg->setReversePathArraySize(reversePathArraySize);
            for (uint i=0; i<reversePathArraySize; i++)
                responseMsg->setReversePath(i, msg->getReversePath(i));
            responseMsg->setID(OverlayKey::random());
            responseMsg->setSearchHopCount(reversePathArraySize);
            responseMsg->setLength(SEARCHRESPONSE_L(responseMsg));

            forwardSearchResponseMessage(responseMsg);
        }
    }
}


void Gia::processSearchMessage(SearchMessage* msg, bool fromApplication)
{
    OverlayKey searchKey = msg->getSearchKey();

    if (keyList.contains(searchKey)) {
        // this node contains search key
        sendSearchResponseMessage(thisNode, msg);
    }

    // check if neighbors contain search key
    for (uint i = 0; i < neighbors->getSize(); i++) {
        KeyList* keyList = neighbors->getNeighborKeyList(i);
        if (keyList->contains(searchKey))
            sendSearchResponseMessage(*(neighbors->get
                                        (i)), msg);
    }

    // forward search-message to next hop
    if (msg->getMaxResponses() > 0) {
        // actualize reverse path
        uint reversePathSize = msg->getReversePathArraySize();

        if (optimizeReversePath)
            for (uint i=0; i<reversePathSize; i++) {
                if (msg->getReversePath(i) == thisNode.getNodeHandle().key) {
                    // Our node already in ReversePath.
                    // Delete successor nodes from ReversePath
                    msg->setLength(msg->length() - (reversePathSize - i)*KEY_L);
                    reversePathSize = i; // set new array size
                    break;
                }
            }

        msg->setReversePathArraySize(reversePathSize+1);
        msg->setReversePath(reversePathSize, thisNode.getNodeHandle().key);
        msg->setLength(msg->length() + KEY_L);

        forwardMessage(msg, fromApplication);
    } else {
        tokenFactory->grantToken();
        delete msg;
    }
}

void Gia::deliverSearchResult(SearchResponseMessage* msg)
{
    OverlayCtrlInfo* overlayCtrlInfo = new OverlayCtrlInfo();
    overlayCtrlInfo->setHopCount(msg->getSearchHopCount());
    overlayCtrlInfo->setSrcNode(msg->getSrcNode());

    GIAanswer* deliverMsg = new GIAanswer("GIA-Answer");
    deliverMsg->setCommand(GIA_ANSWER);
    deliverMsg->setControlInfo(overlayCtrlInfo);
    deliverMsg->setSearchKey(msg->getSearchKey());
    deliverMsg->setNode(msg->getFoundNode().getNodeHandle());
    deliverMsg->setLength(GIAGETRSP_L(msg));

    send(deliverMsg, "to_app");
    if (debugOutput)
        EV << "(Gia) Deliver search response " << msg
        << " to application at " << thisNode << endl;

    delete msg;
}

void Gia::finishOverlay()
{
    // statistics
    recordScalar("GIA: JOIN-Messages Count", stat_joinCount);
    recordScalar("GIA: JOIN Bytes sent", stat_joinBytesSent);
    recordScalar("GIA: JOIN::REQ Messages", stat_joinREQ);
    recordScalar("GIA: JOIN::REQ Bytes sent", stat_joinREQBytesSent);
    recordScalar("GIA: JOIN::RSP Messages", stat_joinRSP);
    recordScalar("GIA: JOIN::RSP Bytes sent", stat_joinRSPBytesSent);
    recordScalar("GIA: JOIN::ACK Messages", stat_joinACK);
    recordScalar("GIA: JOIN::ACK Bytes sent", stat_joinACKBytesSent);
    recordScalar("GIA: JOIN::DNY Messages", stat_joinDNY);
    recordScalar("GIA: JOIN::DNY Bytes sent", stat_joinDNYBytesSent);
    recordScalar("GIA: DISCONNECT:IND Messages", stat_disconnectMessages);
    recordScalar("GIA: DISCONNECT:IND Bytes sent",
		 stat_disconnectMessagesBytesSent);
    recordScalar("GIA: UPDATE:IND Messages", stat_updateMessages);
    recordScalar("GIA: UPDATE:IND Bytes sent", stat_updateMessagesBytesSent);
    recordScalar("GIA: TOKEN:IND Messages", stat_tokenMessages);
    recordScalar("GIA: TOKEN:IND Bytes sent", stat_tokenMessagesBytesSent);
    recordScalar("GIA: KEYLIST:IND Messages", stat_keyListMessages);
    recordScalar("GIA: KEYLIST:IND Bytes sent", stat_keyListMessagesBytesSent);
    recordScalar("GIA: ROUTE:IND Messages", stat_routeMessages);
    recordScalar("GIA: ROUTE:IND Bytes sent", stat_routeMessagesBytesSent);
    recordScalar("GIA: Neighbors max", stat_maxNeighbors);
    recordScalar("GIA: Neighbors added", stat_addedNeighbors);
    recordScalar("GIA: Neighbors removed", stat_removedNeighbors);
    recordScalar("GIA: Level of satisfaction messages ",
                 stat_numSatisfactionMessages);
    recordScalar("GIA: Level of satisfaction max ",stat_maxLevelOfSatisfaction);
    recordScalar("GIA: Level of satisfaction avg ",
                 stat_sumLevelOfSatisfaction / stat_numSatisfactionMessages);
    bootstrapOracle->removePeer(thisNode.getNodeHandle());
}

Gia::~Gia()
{
    cancelAndDelete(satisfaction_timer);
    cancelAndDelete(update_timer);
    cancelAndDelete(timedoutMessages_timer);
    cancelAndDelete(timedoutNeighbors_timer);
    cancelAndDelete(sendKeyList_timer);
    delete msgBookkeepingList;
}

