//
// Copyright (C) 2006 Institut fuer Telematik, Universitaet Karlsruhe (TH)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file Neighbors.h
 * @author Robert Palmer
 */

#ifndef __NEIGHBORS_H_
#define __NEIGHBORS_H_

#include <vector>

#include <omnetpp.h>

#include <OverlayKey.h>
#include <InitStages.h>

#include "GiaNode.h"
#include "NeighborNode.h"


/**
 * This class is for managing all neighbor nodes
 */
class Neighbors : public cSimpleModule
{
  public:
    // OMNeT++ methodes
    /**
     * Sets init stage
     */
    virtual int numInitStages() const
    {
        return MAX_STAGE_OVERLAY + 1;
    }

    /**
     * Initializes this class and set some WATCH(variable) for OMNeT++
     * @param stage Level of initialization (OMNeT++)
     */
    virtual void initialize( int stage );

    /**
     * This module doesn't handle OMNeT++ messages
     * @param msg OMNeT++ message
     */
    virtual void handleMessages( cMessage* msg );

    /**
     * Save statistical data
     */
    virtual void finish();

    // class methodes
    /**
     * @return Number of neighbors
     */
    virtual uint getSize();

    /**
     * @param node GiaNode to check
     * @return true if node is our neighbor
     */
    virtual bool contains(const GiaNode& node) const;

    /**
     * @param key to check
     * @return true if node with corresponding key is our neighbor
     */
    virtual bool contains(const OverlayKey& key) const;

    /**
     * Returns position of node 
     * @param key to check
     * @return -1 if neighbor-list doesn't contain node, else position is returned
     */
    virtual int getPosition(const OverlayKey& key) const;

    /**
     * Returns position of node 
     * @param node to check
     * @return -1 if neighbor-list doesn't contain node, else position is returned
     */
    virtual int getPosition(const GiaNode& node) const;

    /**
     * Adds a new neighbor to our neighbor list
     * @param node New neighbor to add
     */
    virtual void add
        (const GiaNode& node);

    /**
     * Removes neighbor from our neighbor list
     * @param node Node to remove to
     */
    virtual void remove
        (const GiaNode& node);

    /**
     * Removes neighbor from neighbor list at given position
     * @param position 
     */
    virtual void remove
        (int position);

    /**
     * Get neighbor at position
     * @param position
     * @return GiaNode
     */
    virtual GiaNode* get
        (int position);

    /**
     * Get neighbor from neighborlist
     * @param node GiaNode
     * @return GiaNode
     */
    virtual GiaNode* get
        (const GiaNode& node);

    /**
     * Get GiaNode from neighborlist
     * @param key GiaNode-Key
     * @return GiaNode
     */
    virtual GiaNode* get
        (const OverlayKey& key);

    /**
     * Update timestamp
     */
    void updateTimestamp(const GiaNode& node);

    /**
     * Removes timedout nodes
     */
    void removeTimedoutNodes();

    /**
     * Sets the keyList of neighbor at position pos
     * @param pos Position in vector
     * @param keyList KeyList to set
     */
    void setNeighborKeyList(uint pos, const KeyList& keyList);

    /**
     * Get keylist from neighbor
     * @param pos Position of neighbor
     * @return keylist of neighbor
     */
    KeyList* getNeighborKeyList(uint pos);

  protected:
    std::vector<NeighborNode> neighbors; /**< contains all current neighbors */
    GiaNode thisNode; /** this node */
    simtime_t timeout; /** timeout for neighbors */
    GiaNode unspecNode; /** an unspecified node */
};

#endif
