//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file IPv4UnderlayConfigurator.cc
 * @author Markus Mauch, Stephan Krause, Bernhard Heep
 */


#include <vector>

#include <omnetpp.h>

#include "AccessNet.h"

#include "IPv4UnderlayConfigurator.h"


Define_Module(IPv4UnderlayConfigurator);

void IPv4UnderlayConfigurator::initializeUnderlay(int stage)
{
    //backbone configuration
    if(stage == MIN_STAGE_UNDERLAY) {
        // Find all router module types.
        cTopology topo("topo");
        const char* typeNames[6];
        typeNames[0] = "Router";
        typeNames[1] = "OverlayRouter";
        typeNames[2] = "AccessRouter";
        typeNames[3] = "OverlayAccessRouter";
        typeNames[4] = "TunOutRouter";
        typeNames[5] = NULL;
        topo.extractByModuleType(typeNames);

        // Assign IP addresses to all router modules.
        std::vector<uint32> nodeAddresses;
        nodeAddresses.resize(topo.nodes());

	// IP addresses for backbone
	// Take start IP from config file
	// FIXME: Make Netmask for Routers configurable!
        uint32 lowIPBoundary = IPAddress(par("startIP").stringValue()).getInt();

        // uint32 lowIPBoundary = uint32((1 << 24) + 1);
        int numIPNodes = 0;				
        for (int i = 0; i < topo.nodes(); i++) {
            ++numIPNodes;

            uint32 addr = lowIPBoundary + uint32(numIPNodes << 16);
            nodeAddresses[i] = addr;

            // update ip display string
            if (ev.isGUI()) {
                const char* ip_disp = const_cast<char*>(IPAddress(addr).str().c_str());
                topo.node(i)->module()->displayString().insertTag("t", 0);
                topo.node(i)->module()->displayString().setTagArg("t", 0, ip_disp);
                topo.node(i)->module()->displayString().setTagArg("t", 1, "l");
                topo.node(i)->module()->displayString().setTagArg("t", 2, "red");
            }

            // find interface table and assign address to all (non-loopback) interfaces
            InterfaceTable* ift = IPAddressResolver().interfaceTableOf(topo.node(i)->module());

            for ( int k = 0; k < ift->numInterfaces(); k++ ) {
                InterfaceEntry* ie = ift->interfaceAt(k);
                if (!ie->isLoopback()) {
                    ie->ipv4()->setInetAddress(IPAddress(addr));
                    // full address must match for local delivery
                    ie->ipv4()->setNetmask(IPAddress::ALLONES_ADDRESS);
                }
            }
        }

        // Fill in routing tables.
        for ( int i = 0; i < topo.nodes(); i++ ) {
            cTopology::Node* destNode = topo.node(i);
            uint32 destAddr = nodeAddresses[i];

            // calculate shortest paths from everywhere towards destNode
            topo.unweightedSingleShortestPathsTo(destNode);
	    
	    // If destNode is the outRouter, add a default route
	    // to outside network via the TunOutDevice and a route to the
	    // Gateway
	    if ( strcmp(destNode->module()->name(), "outRouter" ) == 0 ) {
		RoutingEntry* defRoute = new RoutingEntry();
		defRoute->host = IPAddress::UNSPECIFIED_ADDRESS;
		defRoute->netmask = IPAddress::UNSPECIFIED_ADDRESS;
		defRoute->gateway = IPAddress(par("gatewayIP").stringValue());
		defRoute->interfaceName = "tunDev";
		defRoute->interfacePtr = IPAddressResolver().interfaceTableOf(destNode->module())->
		    interfaceByName("tunDev");
		defRoute->type = RoutingEntry::REMOTE;
		defRoute->source = RoutingEntry::MANUAL;
		IPAddressResolver().routingTableOf(destNode->module())->addRoutingEntry(defRoute);

		RoutingEntry* gwRoute = new RoutingEntry();
		gwRoute->host = IPAddress(par("gatewayIP").stringValue());
                gwRoute->netmask = IPAddress(255, 255, 255, 255);
		gwRoute->interfaceName = "tunDev";
		gwRoute->interfacePtr = IPAddressResolver().interfaceTableOf(destNode->module())->
		    interfaceByName("tunDev");
		gwRoute->type = RoutingEntry::DIRECT;
		gwRoute->source = RoutingEntry::MANUAL;
		IPAddressResolver().routingTableOf(destNode->module())->addRoutingEntry(gwRoute);
	    }

	    // add route (with host=destNode) to every routing table in the network
	    for ( int j = 0; j < topo.nodes(); j++ ) {
		// continue if same node
                if ( i == j )
                    continue;

                // cancel simulation if node is not conencted with destination
                cTopology::Node* atNode = topo.node(j);
                if(atNode->paths() == 0)
                    error((std::string(atNode->module()->name()) +
                           ": Network is not entirely connected."
			   "Please increase your value for the "
			   "connectivity parameter").c_str());

                //
                // Add routes at the atNode.
                //

                // find atNode's interface and routing table
                InterfaceTable* ift = IPAddressResolver().interfaceTableOf(atNode->module());
                RoutingTable* rt = IPAddressResolver().routingTableOf(atNode->module());

                // find atNode's interface entry for the next hop node
                int outputGateId = atNode->path(0)->localGate()->id();
                InterfaceEntry *ie = ift->interfaceByNodeOutputGateId(outputGateId);

                // find the next hop node on the path towards destNode
                cModule* next_hop = atNode->path(0)->remoteNode()->module();
                IPAddress next_hop_ip = IPAddressResolver().addressOf(next_hop).get4();


                // Requirement 1: Each router has exactly one routing entry
                // (netmask 255.255.0.0) to each other router
                RoutingEntry* re = new RoutingEntry();

                re->host = IPAddress(destAddr);
                re->interfaceName = ie->name();
                re->interfacePtr = ie;
                re->source = RoutingEntry::MANUAL;
                re->netmask = IPAddress(255, 255, 0, 0);
                re->gateway = IPAddress(next_hop_ip);
                re->type = RoutingEntry::REMOTE;

                rt->addRoutingEntry(re);

                // Requirement 2: Each router has a point-to-point routing
                // entry (netmask 255.255.255.255) for each immediate neighbour
                if ( atNode->distanceToTarget() == 1 ) {
                    RoutingEntry* re2 = new RoutingEntry();

                    re2->host = IPAddress(destAddr);
                    re2->interfaceName = ie->name();
                    re2->interfacePtr = ie;
                    re2->source = RoutingEntry::MANUAL;
                    re2->netmask = IPAddress(255, 255, 255, 255);
                    re2->type = RoutingEntry::DIRECT;

                    rt->addRoutingEntry(re2);
                }

		// If destNode is the outRouter, add a default route
		// to the next hop in the direction of the outRouter
		if ( strcmp(destNode->module()->name(), "outRouter" ) == 0 ) {
		    RoutingEntry* defRoute = new RoutingEntry();
		    defRoute->host = IPAddress::UNSPECIFIED_ADDRESS;
		    defRoute->netmask = IPAddress::UNSPECIFIED_ADDRESS;
		    defRoute->gateway = IPAddress(next_hop_ip);
		    defRoute->interfaceName = ie->name();
		    defRoute->interfacePtr = ie;
		    defRoute->type = RoutingEntry::REMOTE;
		    defRoute->source = RoutingEntry::MANUAL;
		    
		    rt->addRoutingEntry(defRoute);
		}
	    }
        }
    }
    //accessnet configuration
    else if(stage == MAX_STAGE_UNDERLAY) {
        // fetch some parameters
        accessRouterNum = parentModule()->par("accessRouterNum");
        overlayAccessRouterNum = parentModule()->par("overlayAccessRouterNum");

        // flag indicating simulation initialization phase (true) vs. normal mode (false)
        init = true;

        // count the overlay clients
        overlayTerminalCount = 0;
        numCreated = 0;
        numKilled = 0;

        // add access node modules to access node vector
        // and assing the channel tpye to be used by the access node
        cModule* node;
        AccessNet* nodeAccess;
        for ( int i = 0; i < accessRouterNum; i++ ) {
            node = parentModule()->submodule("accessRouter", i);
            accessNode.push_back( node );
            nodeAccess = check_and_cast<AccessNet*>( node->submodule("accessNet") );
            nodeAccess->selectChannel( channelTypes[intuniform(0, channelTypes.size()-1)] );
        }

        for ( int i = 0; i < overlayAccessRouterNum; i++ ) {
            node = parentModule()->submodule("overlayAccessRouter", i);
            accessNode.push_back( node );
            nodeAccess = check_and_cast<AccessNet*>( node->submodule("accessNet") );
            nodeAccess->selectChannel( channelTypes[intuniform(0, channelTypes.size()-1)] );
        }

        // add the overlay nodes
	firstNodeId = createRandomNode(true);
        for ( int i = 1; i < initialOverlayTerminalNum; i++ ) {
            createRandomNode(true);
        }

        // debug stuff
        WATCH_PTRVECTOR(accessNode);

        // update display
        setDisplayString();

        // initialize simulation
        if ( par("simulateMobility") ) {
            cMessage* msg = new cMessage();
            scheduleAt(simulation.simTime(), msg);
        }
    }
}

int IPv4UnderlayConfigurator::createRandomNode(bool initialize)
{
    // derive overlay node from ned
    cModuleType* moduleType = findModuleType("OverlayHost");//parentModule()->par("overlayTerminalType"));
    cModule* node = moduleType->create("overlayTerminal", parentModule());

    node->par("overlayType").setStringValue(parentModule()->par("overlayType"));
    node->par("tier2Type").setStringValue(parentModule()->par("tier1Type"));
    node->par("tier2Type").setStringValue(parentModule()->par("tier2Type"));
    node->par("tier3Type").setStringValue(parentModule()->par("tier3Type"));


    node->setGateSize("in", 1);
    node->setGateSize("out", 1);
    node->setDisplayString("i=device/wifilaptop_l;i2=block/circle_s");
    node->buildInside();
    node->scheduleStart(simulation.simTime());

    // add node to a randomly chosen access net
    AccessNet* accessNet = check_and_cast<AccessNet*>
                           (accessNode[intuniform(0, accessNode.size() - 1)]->submodule("accessNet"));
    accessNet->addOverlayNode(node);

    // append index to module name
    char buf[80];
    sprintf(buf, "overlayTerminal[%i]", numCreated);
    node->setName(buf);

    // if the node was not created during startup we have to
    // finish the initialization process manually
    if ( !initialize) {
        for (int i = MAX_STAGE_UNDERLAY + 1; i < NUM_STAGES_ALL; i++) {
            node->callInitialize(i);
        }
    }

    overlayTerminalCount++;
    numCreated++;

    return node->id();
}

//TODO: getRandomNode()
void IPv4UnderlayConfigurator::killRandomNode()
{
    // find an access net with one or more overlay nodes
    AccessNet* accessNetModule;
    do {
        int z = intuniform(0, accessNode.size()-1);
        accessNetModule = check_and_cast<AccessNet*>(accessNode[z]->submodule("accessNet"));
    } while ( accessNetModule->size() == 0 );

    // remove a random node from the access net and delete it
    int index = intuniform(0, accessNetModule->size() - 1);
    
    //TODO: keepFirstNode in while-loop?
    if(keepFirstNode && (firstNodeId == accessNetModule->getOverlayNodeId(index)))
      return;

    cModule* node = accessNetModule->removeOverlayNode(index);//intuniform(0, accessNetModule->size() - 1));
    node->callFinish();
    node->deleteModule();

    overlayTerminalCount--;
    numKilled++;
}

void IPv4UnderlayConfigurator::migrateRandomNode()
{
    // find an access net with one or more overlay nodes
    AccessNet* accessNetModule;
    do {
        int z = intuniform(0, accessNode.size()-1);
        accessNetModule = check_and_cast<AccessNet*>(accessNode[z]->submodule("accessNet"));
    } while ( accessNetModule->size() == 0 );

    // disconnect a random overlay node
    int index = intuniform(0, accessNetModule->size() - 1);
    
    if(keepFirstNode && (firstNodeId == accessNetModule->getOverlayNodeId(index)))
      return;

    cModule* node = accessNetModule->removeOverlayNode(index);//intuniform(0, accessNetModule->size() - 1));

    node->bubble("I am migrating!");

    // connect the node to another access net
    AccessNet* newAccessNetModule;
    do {
        newAccessNetModule = check_and_cast<AccessNet*>(accessNode[intuniform(0, accessNode.size() - 1)]->submodule("accessNet"));

    } while((newAccessNetModule == accessNetModule) && (accessNode.size() != 1));

    newAccessNetModule->addOverlayNode(node, true);

    // inform the notofication board about the migration
    NotificationBoard* nb = check_and_cast<NotificationBoard*>(node->submodule("notificationBoard"));
    nb->fireChangeNotification(NF_HOSTPOSITION_UPDATED);
}

void IPv4UnderlayConfigurator::setDisplayString()
{
    char buf[80];
    sprintf(buf, "%i overlay clients\n%i access router\n%i overlay access router", overlayTerminalCount, accessRouterNum, overlayAccessRouterNum);
    displayString().setTagArg("t", 0, buf);
}

void IPv4UnderlayConfigurator::finish()
{
    // statistics
    recordScalar("Terminals added", numCreated);
    recordScalar("Terminals removed", numKilled);

    struct timeval now, diff;
    gettimeofday(&now, NULL);
    timersub(&now, &initFinishedTime, &diff);
    printf("Simulation time: %li.%06li\n", diff.tv_sec, diff.tv_usec);
}
