//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file SimpleNetConfigurator.cc
 * @author Stephan Krause
 * @author Bernhard Heep (migrateRandomNode)
 */

#include <omnetpp.h>

#include "GlobalRoutingHashMap.h"
#include "InterfaceTable.h"
#include "InterfaceEntry.h"
#include "IPv4InterfaceData.h"
#include "IPAddressResolver.h"

#include "SimpleUDP.h"

#include "SimpleNetConfigurator.h"


Define_Module(SimpleNetConfigurator);

void SimpleNetConfigurator::initializeUnderlay(int stage)
{
    if(stage != MAX_STAGE_UNDERLAY)
        return;

    // fetch some parameters
    routingHashMap = GlobalRoutingHashMapAccess().get();
    moduleType = findModuleType(par("overlayTerminalType"));

    // set maximum coordinates and send queue length
    SimpleNodeEntry::setFieldSize(par("fieldSize"));
    SimpleNodeEntry::setSendQueueLength(par("sendQueueLength"));

    // FIXME get address from parameter
    nextFreeAddress = 0x1000001;

    // flag indicating simulation initialization phase (true)
    // vs. normal mode (false)
    init = true;

    // count the overlay clients
    overlayTerminalCount = 0;
    numCreated = 0;
    numKilled = 0;

    // add the overlay nodes
    firstNodeId = createRandomNode(true);
    for (int i = 1; i < initialOverlayTerminalNum; i++) {
        createRandomNode(true);
    }

    // update display
    setDisplayString();

    // initialize simulation
    if (par("simulateMobility")) {
        cMessage* msg = new cMessage();
        scheduleAt(simulation.simTime(), msg);
    }
}

int SimpleNetConfigurator::createRandomNode(bool initialize)
{
    // derive overlay node from ned
    cModule* node = moduleType->create("overlayTerminal", parentModule());

    node->par("overlayType").setStringValue(parentModule()->par("overlayType"));
    node->par("tier1Type").setStringValue(parentModule()->
					       par("tier1Type"));
    node->par("tier2Type").setStringValue(parentModule()->
					       par("tier2Type"));
    node->par("tier3Type").setStringValue(parentModule()->
					       par("tier3Type"));

    node->setDisplayString("i=device/wifilaptop_l;i2=block/circle_s");
    node->buildInside();
    node->scheduleStart(simulation.simTime());

    for (int i = 0; i < MAX_STAGE_UNDERLAY + 1; i++) {
        node->callInitialize(i);
    }

    // FIXME use only IPv4?
    IPvXAddress addr = IPAddress(nextFreeAddress++);

    SimpleNodeEntry entry(node, findChannelType(channelTypes[
			    intuniform(0, channelTypes.size() - 1)].c_str()));

    routingHashMap->insertNode(addr, entry);
    SimpleUDP* simple = check_and_cast<SimpleUDP*>(node->submodule("udp"));
    simple->setNodeEntry(entry);

    // Add pseudo-Interface to node's interfaceTable
    IPv4InterfaceData* ifdata = new IPv4InterfaceData;
    ifdata->setInetAddress(addr.get4());
    ifdata->setNetmask(IPAddress("255.255.255.255"));
    InterfaceEntry* e = new InterfaceEntry;
    e->setName("dummy interface");
    e->setIPv4Data(ifdata);

    IPAddressResolver().interfaceTableOf(node)->addInterface(e, NULL);

    // append index to module name
    char buf[80];
    sprintf(buf, "overlayTerminal[%i]", numCreated);
    node->setName(buf);

    // if the node was not created during startup we have to
    // finish the initialization process manually
    if ( !initialize) {
        for (int i = MAX_STAGE_UNDERLAY + 1; i < NUM_STAGES_ALL; i++) {
            node->callInitialize(i);
        }
    }

    //show ip... todo: migrate
    /*
    if (ev.isGUI()) {
    node->displayString().insertTag("t", 0);
    node->displayString().setTagArg("t", 0, addr.str().c_str());
    node->displayString().setTagArg("t", 1, "l");
    }
    */

    overlayTerminalCount++;
    numCreated++;

    return node->id();
}

//TODO: getRandomNode()
void SimpleNetConfigurator::killRandomNode()
{
    // FIXME: better way to do this?
    // getEntry is inefficient!
    const GlobalRoutingHashMap::RouteEntry e = routingHashMap->getEntry(
                intuniform( 0, routingHashMap->size() -1 ));
    cGate* gate = e.second.getGate();

    cModule* node = gate->ownerModule()->parentModule();

    if(keepFirstNode && (node->id() == firstNodeId))
      return;

    InterfaceEntry* ie = IPAddressResolver().interfaceTableOf(node)->
                         interfaceByName("dummy interface");

    delete ie->ipv4();

    node->callFinish();
    node->deleteModule();
    routingHashMap->removeNode( e.first );

    overlayTerminalCount--;
    numKilled++;
}

void SimpleNetConfigurator::migrateRandomNode()
{
    GlobalRoutingHashMap::RouteEntry e = routingHashMap->getEntry(
                                             intuniform(0, routingHashMap->size() -1));

    cGate* gate = e.second.getGate();
    cModule* node = gate->ownerModule()->parentModule();

    if(keepFirstNode && (node->id() == firstNodeId))
      return;

    routingHashMap->removeNode(e.first);

    node->bubble("I am migrating!");

    // FIXME use only IPv4?
    IPvXAddress addr = IPAddress(nextFreeAddress++);

    SimpleNodeEntry entry(node, findChannelType(channelTypes[intuniform(
                                   0, channelTypes.size() - 1)].c_str()));

    routingHashMap->insertNode(addr, entry);
    SimpleUDP* simple = check_and_cast<SimpleUDP*>(gate->ownerModule());
    simple->setNodeEntry(entry);

    InterfaceEntry* ie = IPAddressResolver().interfaceTableOf(node)->
                         interfaceByName("dummy interface");

    delete ie->ipv4();

    // Add pseudo-Interface to node's interfaceTable
    IPv4InterfaceData* ifdata = new IPv4InterfaceData;
    ifdata->setInetAddress(addr.get4());
    ifdata->setNetmask(IPAddress("255.255.255.255"));
    ie->setIPv4Data(ifdata);

    // inform the notofication board about the migration
    NotificationBoard* nb = check_and_cast<NotificationBoard*>
                            (node->submodule("notificationBoard"));
    nb->fireChangeNotification(NF_HOSTPOSITION_UPDATED);
}


void SimpleNetConfigurator::setDisplayString()
{
    //
    // Updates the statistics display string.
    //

    char buf[80];
    sprintf(buf, "%i overlay clients", overlayTerminalCount);
    displayString().setTagArg("t", 0, buf);
}

void SimpleNetConfigurator::finish()
{
    // statistics
    recordScalar("Terminals added", numCreated);
    recordScalar("Terminals removed", numKilled);

    struct timeval now, diff;
    gettimeofday(&now, NULL);
    timersub(&now, &initFinishedTime, &diff);
    printf("Simulation time: %li.%06li\n", diff.tv_sec, diff.tv_usec);
}
