//
// Copyright (C) 2004 Andras Varga
// Copyright (C) 2000 Institut fuer Telematik, Universitaet Karlsruhe
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
//

/**
 * @file SimpleNodeEntry.cc
 * @author Bernhard Heep
 */

#include <sstream>

#include "SimpleNodeEntry.h"


uint SimpleNodeEntry::fieldSize;
uint SimpleNodeEntry::sendQueueLength;

SimpleNodeEntry::SimpleNodeEntry(cModule* node,
                                 cChannelType* type)
{
    ingate = node->submodule("udp")->gate("network_in");

    x = uniform(0, fieldSize);
    y = uniform(0, fieldSize);

    cBasicChannel* temp = dynamic_cast<cBasicChannel*>(type->create("temp"));

    bandwidth = temp->datarate();
    errorRate = temp->error();
    accessDelay = temp->delay();

    txMaxQueueTime = ((float)sendQueueLength * 8) / bandwidth;
    txFinished = simulation.simTime(); // 0?

    delete temp;
}

float SimpleNodeEntry::operator-(const SimpleNodeEntry& entry) const
{
    return sqrt(pow(x - entry.x, 2) + pow(y - entry.y, 2));
}

SimpleNodeEntry::SimpleDelay SimpleNodeEntry::calcDelay(const SimpleUDPPacket& msg, const SimpleNodeEntry& dest)
{
    if (uniform(0, 1) < errorRate || uniform(0, 1) < dest.errorRate)
        return SimpleDelay(0, false);

    simtime_t now = simulation.simTime();
    simtime_t bandwidthDelay= ((msg.byteLength() * 8) / bandwidth);
    simtime_t newTxFinished = fmax(txFinished, now) + bandwidthDelay;

    // send queue
    if((newTxFinished > now + txMaxQueueTime) && (txMaxQueueTime != 0)) {
        EV << "(SimpleNodeEntry::calcDelay()) send queue overrun!\n"
        << "                  newTxFinished = fmax(txFinished, now) + bandwidthDelay\n"
        << "                  newTxFinished = " << newTxFinished
        << "\n                  txFinished = " << txFinished
        << "\n                  now = " << now
        << "\n                  bandwidthDelay = " << bandwidthDelay
        << "\n                  (newTxFinished > now + txMaxQueueTime) == true"
        << "\n                  txMaxQueueTime = " << txMaxQueueTime << endl;

        return SimpleDelay(0, false);
    }

    txFinished = newTxFinished;

    simtime_t destBandwidthDelay = (msg.byteLength() * 8) / dest.bandwidth;
    simtime_t coordDelay = 0.001 * (*this - dest);

    return SimpleDelay(txFinished + coordDelay + destBandwidthDelay - now, true);
}

std::string SimpleNodeEntry::info() const
{
    std::ostringstream str;
    str << *this;
    return str.str();
}

std::ostream& operator<<(std::ostream& out, const SimpleNodeEntry& entry)
{
    out << "(x:" << entry.x << ", y:" << entry.y
    << ")\nbandwidth = " << entry.bandwidth
    << ",\ndelay = " << entry.accessDelay
    << ",\ntxMaxQueueTime = " << entry.txMaxQueueTime
    << ",\ntxFinished = " << entry.txFinished;

    return out;
}
