#include <netinet/ip.h>
#include <netinet/udp.h>
#include "IPDatagram_m.h"
#include "UDPPacket.h"

#include "TunOutDevice.h"


Define_Module(TunOutDevice);

char* TunOutDevice::encapsulate(cMessage *msg, unsigned int* length)
{
    struct udppseudohdr {
        uint32_t saddr;
        uint32_t daddr;
        uint8_t zero;
        uint8_t protocol;
        uint16_t lenght;
    }
    * pseudohdr;

    unsigned int payloadlen;
    static unsigned int iplen = 20; // we don't generate IP options
    static unsigned int udplen = 8;
    cMessage* payloadMsg = NULL;
    char* buf = NULL, *payload = NULL;
    uint32_t saddr, daddr; 
    iphdr* ip_buf;
    udphdr* udp_buf;

    IPDatagram* IP = check_and_cast<IPDatagram*>(msg);
    // FIXME: Cast ICMP-Messages
    UDPPacket* UDP = dynamic_cast<UDPPacket*>(IP->decapsulate());
    if (!UDP) {
	EV << "Can't parse non-UDP packets (e.g. ICMP) (yet)...\n";
	goto parse_error;
    }
    payloadMsg = UDP->decapsulate();

    // parse payload
    payload = parser->encapsulatePayload(payloadMsg, &payloadlen);
    if (!payload )
	goto parse_error;

    *length = payloadlen + iplen + udplen;
    if( *length > mtu ) {
        EV << "TunOutDevice::encapsulate: Error: Packet too big! Size = " << *length << " MTU = " << mtu << "\n";
	goto parse_error;
    }

    buf = new char[*length];

    // We use the buffer to build an ip packet.
    // To minimise unnecessary copying, we start with the payload
    // and write it to the end of the buffer
    memcpy( (buf + iplen + udplen), payload, payloadlen);

    // write udp header in front of the payload
    udp_buf = (udphdr*) (buf + iplen);
    udp_buf->source = htons(UDP->sourcePort());
    udp_buf->dest = htons(UDP->destinationPort());
    udp_buf->len = htons(udplen + payloadlen);
    udp_buf->check = 0;

    // Write udp pseudoheader in from of udp header
    // this will be overwritten by ip header
    pseudohdr = (udppseudohdr*) (buf + iplen - sizeof(struct udppseudohdr));
    saddr = htonl(IP->srcAddress().getInt());
    daddr = htonl(IP->destAddress().getInt());
    pseudohdr->saddr = saddr;
    pseudohdr->daddr = daddr;
    pseudohdr->zero = 0;
    pseudohdr->protocol = IPPROTO_UDP;
    pseudohdr->lenght = udp_buf->len;

    // compute UDP checksum
    udp_buf->check = cksum((uint16_t*) pseudohdr, sizeof(struct udppseudohdr) + udplen + payloadlen);

    // write ip header to begin of buffer
    ip_buf = (iphdr*) buf;
    ip_buf->version = 4; // IPv4
    ip_buf->ihl = iplen / 4;
    ip_buf->tos = IP->diffServCodePoint();
    ip_buf->tot_len = htons(*length);
    ip_buf->id = htons(IP->identification());
    ip_buf->frag_off = htons(IP_DF); // DF, no fragments
    ip_buf->ttl = IP->timeToLive();
    ip_buf->protocol = IPPROTO_UDP;
    ip_buf->saddr = saddr;
    ip_buf->daddr = daddr;
    ip_buf->check = 0;
    ip_buf->check = cksum((uint16_t*) ip_buf, iplen);

    delete IP;
    delete UDP;
    delete payloadMsg;
    delete payload;

    return buf;

parse_error:
    delete IP;
    delete UDP;
    delete payloadMsg;
    delete payload;
    return NULL;

}

cMessage* TunOutDevice::decapsulate(char* buf, uint32_t length)
{
    // Message starts with IP header
    iphdr* ip_buf = (iphdr*) buf;
    udphdr* udp_buf;
    IPDatagram* IP = new IPDatagram;
    UDPPacket* UDP = new UDPPacket;
    cMessage* payload = 0;
    unsigned int payloadLen, datagramlen;
    unsigned int packetlen = ntohs(ip_buf->tot_len);

    // Parsing of IP header, sanity checks
    if (  packetlen != length ) {
        ev << "TunOutDevice: Dropping bogus packet, header says: length = " << packetlen << " but actual length = " << length <<".\n";
        goto parse_error;
    }
    if (  packetlen > mtu ) {
        ev << "TunOutDevice: Dropping bogus packet, length = " << packetlen << " but mtu = " << mtu <<".\n";
        goto parse_error;
    }
    if ( ip_buf->version != 4 ) {
        ev << "TunOutDevice: Dropping Packet: Packet is not IPv4.\n";
        goto parse_error;
    }
    if ( ntohs(ip_buf->frag_off) & 0xBFFF ) { // mask DF bit
        ev << "TunOutDevice: Dropping Packet: Can't handle fragmented packets.\n";
        goto parse_error;
    }
    if ( ip_buf->protocol != IPPROTO_UDP ) { // FIXME: allow ICMP packets
        ev << "TunOutDevice: Dropping Packet: Packet is not UDP.\n";
        goto parse_error;
    }
    IP->setSrcAddress( IPAddress( ntohl(ip_buf->saddr) ));
    IP->setDestAddress( IPAddress( ntohl(ip_buf->daddr) ));
    IP->setTransportProtocol( ip_buf->protocol );
    IP->setTimeToLive( ip_buf->ttl );
    IP->setIdentification( ntohs(ip_buf->id) );
    IP->setMoreFragments( false );
    IP->setDontFragment( true );
    IP->setFragmentOffset( 0 );
    IP->setDiffServCodePoint( ip_buf->tos );
    IP->setLength( ip_buf->ihl*32 );
    // FIXME: check IP and UDP checksum...

    // Parse UDP header, sanity checks
    udp_buf = (udphdr*)( ((uint32_t *)ip_buf) + ip_buf->ihl );
    datagramlen = ntohs(udp_buf->len);
    if ( (datagramlen != packetlen - ip_buf->ihl*4) ) {
        ev << "TunOutDevice: Dropping Packet: Bogus UDP datagram length: len = " << datagramlen << " packetlen = " << packetlen << " ihl*4 " << ip_buf->ihl*4 << ".\n";
        goto parse_error;
    }
    UDP->setSourcePort( ntohs( udp_buf->source ));
    UDP->setDestinationPort( ntohs( udp_buf->dest ));
    UDP->setByteLength( sizeof( struct udphdr ) );

    // parse payload
    payloadLen = datagramlen - sizeof( struct udphdr );
    payload = parser->decapsulatePayload( ((char*) udp_buf) + sizeof( struct udphdr ), payloadLen );
    if (!payload) {
        ev << "TunOutDevice: Parsing of Payload failed, dropping packet.\n";
        goto parse_error;
    }
    // encapsulate messages
    UDP->encapsulate( payload );
    IP->encapsulate( UDP );

    delete buf;
    return IP;

    // In case the parsing of the packet failed, free allocated memory
parse_error:
    delete buf;
    delete IP;
    delete UDP;
    return NULL;
}

