#include "realtimescheduler.h"

inline std::ostream& operator<<(std::ostream& ev, const timeval& tv)
{
    return ev << (unsigned long)tv.tv_sec << "s" << tv.tv_usec << "us";
}

//---

RealtimeScheduler::RealtimeScheduler() : cScheduler()
{
    netw_fd = -1;
    app_fd = -1;
}

RealtimeScheduler::~RealtimeScheduler() { }

void RealtimeScheduler::startRun()
{
    if (initsocketlibonce()!=0)
        throw new cRuntimeError("RealtimeScheduler: Cannot initialize socket library");

    gettimeofday(&baseTime, NULL);

    appModule = NULL;
    appNotificationMsg = NULL;
    module = NULL;
    notificationMsg = NULL;

    if (initializeNetwork()) {
	opp_error("realtimeScheduler error: initializeNetwork failed\n");
    }
}

void RealtimeScheduler::endRun() {}

void RealtimeScheduler::executionResumed()
{
    gettimeofday(&baseTime, NULL);
    baseTime = timeval_substract(baseTime, sim->simTime());
}

void RealtimeScheduler::setInterfaceModule(cModule *mod, cMessage *notifMsg,  PacketBuffer* buffer, int mtu, bool isApp)
{
    if (!mod || !notifMsg || !buffer)
	throw new cRuntimeError("RealtimeScheduler: setInterfaceModule(): arguments must be non-NULL");

    if (!isApp) {
	if (module)
	    throw new cRuntimeError("RealtimeScheduler: setInterfaceModule() already called");
	module = mod;
	notificationMsg = notifMsg;
	packetBuffer = buffer;
	buffersize = mtu;
    } else {
	if (appModule)
	    throw new cRuntimeError("RealtimeScheduler: setInterfaceModule() already called");
	appModule = mod;
	appNotificationMsg = notifMsg;
	appPacketBuffer = buffer;
	appBuffersize = mtu;
    }
}

bool RealtimeScheduler::receiveWithTimeout(long usec)
{
    bool newData = false;
    // prepare sets for select()
    fd_set readFD;
    FD_ZERO(&readFD);

    FD_SET(netw_fd, &readFD);
    if ( app_fd >= 0 ) FD_SET(app_fd, &readFD);

    timeval timeout;
    timeout.tv_sec = 0;
    timeout.tv_usec = usec;

    if (select(FD_SETSIZE, &readFD, NULL, NULL, &timeout) > 0) {
        // Incoming data
	if (FD_ISSET(netw_fd, &readFD)) {
	    char* buf = new char[buffersize];
	    int nBytes = read(netw_fd, buf, buffersize);
	    if (nBytes < 0) {
		opp_error("Read from network device returned an error");
	    } else if (nBytes == 0) {
		opp_error("network device closed...");
	    } else {
		// write data to buffer
		ev << "RealtimeScheduler: received " << nBytes << " bytes\n";
		packetBuffer->push_back(PacketBufferEntry(buf, nBytes));

		// schedule notificationMsg for the interface module
		timeval curTime;
		gettimeofday(&curTime, NULL);
		curTime = timeval_substract(curTime, baseTime);
		simtime_t t = curTime.tv_sec + curTime.tv_usec*1e-6;
		// TBD assert that it's somehow not smaller than previous event's time
		notificationMsg->setArrival(module,-1,t);
		simulation.msgQueue.insert(notificationMsg);
		newData = true;
	    }
	} else if ( (app_fd >= 0) && (FD_ISSET(app_fd, &readFD)) ) {
	    char* buf = new char[appBuffersize];
	    int nBytes = read(app_fd, buf, appBuffersize);
	    if (nBytes < 0) {
		opp_error("Read from network device returned an error");
	    } else if (nBytes == 0) {
		opp_error("network device closed...");
	    } else {
		// write data to buffer
		ev << "RealtimeScheduler: received " << nBytes << " bytes\n";
		appPacketBuffer->push_back(PacketBufferEntry(buf, nBytes));

		// schedule notificationMsg for the interface module
		timeval curTime;
		gettimeofday(&curTime, NULL);
		curTime = timeval_substract(curTime, baseTime);
		simtime_t t = curTime.tv_sec + curTime.tv_usec*1e-6;
		// TBD assert that it's somehow not smaller than previous event's time
		appNotificationMsg->setArrival(appModule,-1,t);
		simulation.msgQueue.insert(appNotificationMsg);
		newData = true;
	    }
	}
    }
    return newData;
}

int RealtimeScheduler::receiveUntil(const timeval& targetTime)
{
    // if there's more than 200ms to wait, wait in 100ms chunks
    // in order to keep UI responsiveness by invoking ev.idle()
    timeval curTime;
    gettimeofday(&curTime, NULL);
    while (targetTime.tv_sec-curTime.tv_sec >=2 ||
	    timeval_diff_usec(targetTime, curTime) >= 200000) {
	if (receiveWithTimeout(100000)) { // 100ms
	    if (ev.idle()) return -1;
	    return 1;
	}
	if (ev.idle()) return -1;
	gettimeofday(&curTime, NULL);
    }

    // difference is now at most 100ms, do it at once
    long usec = timeval_diff_usec(targetTime, curTime);
    if (usec>0)
        if (receiveWithTimeout(usec)) {
	    if (ev.idle()) return -1;
            return 1;
	}
	if (ev.idle()) return -1;
    return 0;
}

cMessage *RealtimeScheduler::getNextEvent()
{
    // assert that we've been configured
    if (!module)
        throw new cRuntimeError("RealtimeScheduler: setInterfaceModule() not called: it must be called from a module's initialize() function");
    if (app_fd >= 0 && !appModule)
        throw new cRuntimeError("RealtimeScheduler: setInterfaceModule() not called from application: it must be called from a module's initialize() function");

    // calculate target time
    timeval targetTime;
    cMessage *msg = sim->msgQueue.peekFirst();
    if (!msg) {
        // if there are no events, wait until something comes from outside
        // TBD: obey simtimelimit, cpu-time-limit
        targetTime.tv_sec = LONG_MAX;
        targetTime.tv_usec = 0;
    } else {
        // use time of next event
        simtime_t eventSimtime = msg->arrivalTime();
        targetTime = timeval_add(baseTime, eventSimtime);
    }

    // if needed, wait until that time arrives
    timeval curTime;
    gettimeofday(&curTime, NULL);
    if (timeval_greater(targetTime, curTime)) {
        int status = receiveUntil(targetTime);
        if (status == -1) {
	    printf("WARNING: receiveUntil returned -1 (user interrupt)\n");
            return NULL; // interrupted by user
	} else if (status == 1) {
            msg = sim->msgQueue.peekFirst(); // received something
	}
    } else {
        //    printf("WARNING: Lagging behind realtime!\n");
        // we're behind -- customized versions of this class may
        // alert if we're too much behind, whatever that means
    }
    // ok, return the message
    return msg;
}

void RealtimeScheduler::sendBytes(const char *buf, size_t numBytes, bool isApp)
{
    if (!buf) {
        ev << "Error sending packet: buf = NULL!\n";
        return;
    }
    if (!isApp) {
	if( numBytes > buffersize ) {
	    ev << "trying to send oversizd packet: size " << numBytes << " mtu " << buffersize << "\n";
	    opp_error("Can't send packet: too large"); //FIXME: Throw exception instead
	}
	write(netw_fd, buf, numBytes);
    } else {
	if( numBytes > appBuffersize ) {
	    ev << "trying to send oversizd packet: size " << numBytes << " mtu " << appBuffersize << "\n";
	    opp_error("Can't send packet: too large"); //FIXME: Throw exception instead
	}
	if ( app_fd < 0 ) opp_error ("Can't send packet to Application: no socket");
	write(app_fd, buf, numBytes);
    }
    // TBD check for errors
}

