#ifndef __REALTIMESCHEDULER_H__
#define __REALTIMESCHEDULER_H__

#include "platdep/time.h"
#include "platdep/sockets.h"
#include <omnetpp.h>
#include <list>

/** This class implements a event scheduler for omnet
 *  It makes the simulation run in realtime (i.e. 1 simsec == 1 sec)
 *  It must be subclassed; its subclasses must handle network
 *  traffic from/to the simulation
 **/
class RealtimeScheduler : public cScheduler
{
public:
    class PacketBufferEntry {
	public:
	    char* data;
	    uint32_t length;
	    PacketBufferEntry(char* buf, uint32_t len) : data(buf), length(len) {};
    };
    typedef std::list<PacketBufferEntry> PacketBuffer;

protected:
    // Buffer, module and FD for network commnication
    int netw_fd;
    cModule *module;
    cMessage *notificationMsg;
    PacketBuffer *packetBuffer;
    size_t buffersize;


    // ... for realworldapp communication
    int app_fd;
    cModule *appModule;
    cMessage *appNotificationMsg;
    PacketBuffer *appPacketBuffer;
    size_t appBuffersize;

    // state
    timeval baseTime;

    /** Initialize the network
     **/
    virtual int initializeNetwork() = 0;

    /**
     * Waits for incoming data on the tun device
     *
     * \param usec Timeout after which to quit waiting (in µsec)
     * \return true if data was read, false otherwise
     **/
    virtual bool receiveWithTimeout(long usec);

    /**
     * Tries to read data until the given time is up
     *
     * \param targetTime stop waiting after this time is up
     * \return 1 if data is read, -1 if there is an error, 0 if no data is read
     **/
    virtual int receiveUntil(const timeval& targetTime);

public:
    /**
     * Constructor.
     */
    RealtimeScheduler();

    /**
     * Destructor.
     */
    virtual ~RealtimeScheduler();

    /**
     * Called at the beginning of a simulation run.
     */
    virtual void startRun();

    /**
     * Called at the end of a simulation run.
     */
    virtual void endRun();

    /**
     * Recalculates "base time" from current wall clock time.
     */
    virtual void executionResumed();

    /**
     * To be called from the module which wishes to receive data from the
     * tun device. The method must be called from the module's initialize()
     * function.
     *
     * \param module Pointer to the module that whishes to receive the data
     * \param notificationMsg A pointer to a message that will be sheduled if there is data to read
     * \param buffer A pointer to the buffer the data will be written into
     * \param mtu Max allowed packet size
     * \param isApp set to "true" if called from a realworldApp
     */
    virtual void setInterfaceModule(cModule *module, cMessage *notificationMsg, PacketBuffer* buffer, int mtu, bool isApp = false);

    /**
     * Scheduler function -- it comes from cScheduler interface.
     */
    virtual cMessage *getNextEvent();

    /**
     * Send data to network
     *
     * \param buf A pointer to the data to be send
     * \param numBytes the length of the data
     * \param isApp set to "true" if called from a realworldApp
     */
    virtual void sendBytes(const char *buf, size_t numBytes, bool isApp = false);
};

#endif

