#!/usr/bin/perl

# Postprocess OverSim output files (.sca or .vec)

use Statistics::Descriptive;

# Print accumulated statistics over all runs
sub printGlobalStatistics {
  if ( $numRuns == 1 ) { 
    print "\n\nOnly one run processed, no global statistics\n"; 
  } else {
    print "\n==========================================================\n";
    print "        Accumulated statistics over $numRuns runs\n";
    print "============================================================\n";
    foreach $key (sort keys(%globalStat)){
      print "$key: Accumulated mean: " . $globalStat{$key}->mean();
      print " (Variance: " . $globalStat{$key}->variance() . ")\n";
    }
  }
}

# Accumulate statistics from multiple runs
sub generateGlobalStatistics {
  foreach $key (keys(%entry)) {
    if (! defined $globalStat{$key} ) { $globalStat{$key} = Statistics::Descriptive::Full->new(); }
    $globalStat{$key}->add_data($entry{$key}->mean());
  }
  $numRuns++;
}

# Output for a single run from a .sca file
sub printScaStats {
  print "\n============================================================\n";
  print "                    $run";
  print "==============================================================\n";
  foreach $key (sort keys(%entry)){
    print "$key: " . $entry{$key}->mean();
    print " (min: " . $entry{$key}->min();
    print ", max: " . $entry{$key}->max() . ")\n";
  }
}

# Processing of a .sca file
sub processSca {
  while(<FILE>){
    if (/^run/) { # New run starts
      $newrun = $_;
      if (keys %entry) {&printScaStats();} # if last run not empty, print stats
      $run = $newrun;
      &generateGlobalStatistics();
      %entry = ();
    } else {
      s/\[\d*\]/\[\]/g; # Delete digits in square brackets
      # Parse lines: 'scalar "<Module>" "<Param>" <Value>'
      /scalar\s*"([^"]*)"\s*("[^"]*")\s*([\d.]*)/;
      $name = "$1 $2";
      if (! defined $entry{$name} ) { $entry{$name} = Statistics::Descriptive::Full->new(); }
      $entry{$name}->add_data($3);
    }
  }
  &printScaStats();
  &generateGlobalStatistics();
  %entry = ();
}

# Output for a single run from a .vec file
sub printVecStats {
  $runNr = $numRuns + 1;
  print "\n=============================================================\n";
  print "                    Run $runNr (file: $filename)\n";
  print "===============================================================\n";
  foreach $key (sort keys(%entry)){
    print "$key: " . $entry{$key}->mean();
    print " (min: " . $entry{$key}->min();
    print ", max: " . $entry{$key}->max();
    print ", 10p: " . $entry{$key}->percentile(10);
    print ", 90p: " . $entry{$key}->percentile(90) . ")\n";
  }
}

# Processing of a .vec file
sub processVec {
  while(<FILE>){
    if (/^vector/) {
      s/\[[0-9]*\]/\[\]/g; # Delete digits in square brackets
      /(\d*)  "(.*)"  (".*")  (\d*)\n$/; # Parse vector definition
      $vectorname[$1] = "$2 $3";

    } elsif (/^([0-9]*)\t([0-9.]*)\t([0-9.]*)\n$/) {
      if (! defined $entry{$vectorname[$1]} ) {
	$entry{$vectorname[$1]} = Statistics::Descriptive::Full->new();
      }
      $entry{$vectorname[$1]}->add_data($3);
    }
  }
  &printVecStats();
  &generateGlobalStatistics();
  %entry = ();
}

## Program starts here
#  Loop over all files
if ( $#ARGV < 0 ) { $ARGV[0] = "-"; }
foreach (@ARGV) {
  open(FILE, $_) || die "Could not open file: $_\n";
  $filename = $_;
  $line = <FILE>;
  $_ = $line;

  # Check whether input is .sca or .vec
  if (/^run/) {
    if ($mode && $mode ne "sca") { 
      die "Error processing $filename: This is a .sca file, previous files were .vec files\n";
    }
    $mode = "sca";
    $run = $line;
    &processSca();

  } elsif (/^vector/) {
    if ($mode && $mode ne "vec") {
      die "Error processing $filename: This is a .vec file, previous files were .sca files\n";
    }
    $mode = "vec";
    s/\[[0-9]*\]/\[\]/g; # Delete digits in square brackets
    /(\d*)  "(.*)"  (".*")  (\d*)\n$/; # Parse vector definition
    $vectorname[$1] = "$2 $3";
    &processVec();

  } else {
    die "Error processing $filename: File type not recognized!\n";
  }
}

&printGlobalStatistics();

